/*********************************************************************
//  created:    2007/11/13 - 10:48
//  filename:   AlascaComponent.h
//
//  author:     Gerald Dherbomez & Sergio Rodriguez
//              Copyright Heudiasyc UMR UTC/CNRS 6599
// 
//  version:    $Id: $
//
//  purpose:    Definition of the AlascaComponent class 
*********************************************************************/

#ifndef ALASCACOMPONENT_H
#define ALASCACOMPONENT_H

#include "Pacpus/kernel/ComponentBase.h"
#include "Pacpus/kernel/DbiteFile.h"
#include "AlascaData.h"

#include <fstream>
#include <QThread>
#include <string>

// Export macro for AlascaXT DLL for Windows only
#ifdef WIN32
#   ifdef ALASCAXT_EXPORTS
        // make DLL
#       define ALASCAXT_API __declspec(dllexport)
#   else
        // use DLL
#       define ALASCAXT_API __declspec(dllimport)
#   endif
#else
    // On other platforms, simply ignore this 
#   define ALASCAXT_API 
#endif

class QEvent;

namespace pacpus {
    
class AlascaSocket;
class ShMem;

struct ScanAlascaObject {
    // todo...
};

// see Manual_Alasca.pdf page 27
struct MessageHeader {
    unsigned long magicWord;
    unsigned long size;
    unsigned long dataType;
    unsigned long timestamp;
};

// see Manual_Alasca.pdf page 28
struct ALASCAXT_API ScanHeader {
    unsigned char version;
    unsigned char scannerType;
    unsigned char ecuId;
    unsigned long timeStamp;
    unsigned short startAngle;
    unsigned short endAngle;
    unsigned short scanCounter;
    unsigned short numPoints;
};

// the class containing Alasca message
class ALASCAXT_API Message
{
public:
    /// Constructor.
    Message()
    {
        time = 0;
        timerange = 0;
        memset(&hMsg,0,sizeof(hMsg));
        memset(&hScan,0,sizeof(hScan));
        memset(body,0,sizeof(body));
    }

    /// Destructor.
    ~Message(){}

    MessageHeader hMsg; // 16 bytes
    ScanHeader hScan;   // 16 bytes
    unsigned char body[ 32 + MAX_SCAN_POINT * 12 ]; // 32 + 12N bytes where N <= 8648 - MH+SH+body
    road_time_t time;
    road_timerange_t timerange;
};

// Structure used to stored alasca data between several decoding processes 
struct MessagePacket {
    road_time_t time;
    std::string data;
    bool previousData;
};

class AlascaDataGenerator; 

class ALASCAXT_API AlascaComponent
        : public QThread
        , public ComponentBase
{
    Q_OBJECT

public:
    AlascaComponent(QString name);
    ~AlascaComponent();

    void run() {}

    virtual void stopActivity(); /*!< to stop the processing thread */
    virtual void startActivity(); /*!< to start the processing thread */
    virtual ComponentBase::COMPONENT_CONFIGURATION configureComponent(XmlComponentConfig config);

    // decoupe le paquet en messages et remplit la liste de 'Message'
    // remplit le message header
    // remplit le scan header
    // détermine si le message est complet (isFull)
    // renvoie le nombre de pending bytes
    void splitPacket(const char * packet, const int length, road_time_t time);

    // decoupe le message en scan
    // remplit la structure alascaData
    // renvoie le nombre de scan (<=8 648)
    unsigned long processMessage(Message & msg);

    // renvoie l'index du magicWord dans le message
    // -1 si aucun magic word trouvé
    long findMagicWord(const char * message, const unsigned length);

    // renvoie la taille du message = 32 + body size
    // 0 si pas trouv�
    long getMessageSize(const char * message, const unsigned length, const long magicWordIndex);

    /// @returns @b true if the message is complete, @b false otherwise
    bool isMessageComplete(const unsigned length, const long size);

public Q_SLOTS:  
    //Fonction qui permet de trier les données reçues et de les enregistrer dans une structure point, puis dans un fichier binaire .dbt
    void customEvent(QEvent * e);
    
public:
    AlascaSocket * A_socket; //on déclare pointeur de socket
    AlascaComponent * myParent;

private:
    std::list<Message> msgList;

    ScanAlascaData alascaData;
    ScanAlascaObject alascaObjects;

    //char * pendingBytes;
    MessagePacket pendingBytes;

    void storePendingBytes(road_time_t time);
    void fillMessageHeader(Message & msg);
    void fillScanHeader(Message & msg);
    short din70000toCentimeters(short n);
    float din70000toMeters(short n);
    // write the data on the disk in the dbt file and the associated binary file
    void writeData();

    // AlascaDataGenerator * generator;

    AlascaXT dbtData_;
    pacpus::DbiteFile dbtFile_;
    std::ofstream dataFile_;

    //ShMem * shmem_;

    // The alasca IP or hostname
    QString host_;

    // The alasca port
    int port_;
};

} // namespace pacpus

#endif // ALASCACOMPONENT_H
