/// Purpose:     This class decodes NMEA0183 data and can store them
///              in dbt files. It implements a thread which gets frames
///              from dataFrame class.
///
/// @date created 2005/09/02 - 8:06
/// @author:      Gerald Dherbomez
/// @version:     $Id: gpsComponent.h 1277 2013-01-10 16:49:06Z bonnetst $
///
/// @todo replace the static shared memory by the class ShMem

#ifndef GPSCOMPONENT_H
#define GPSCOMPONENT_H

#include "Pacpus/kernel/pacpus.h"
#include "Pacpus/kernel/ComponentBase.h"
#include "Pacpus/kernel/DbiteFile.h"
#include "../NMEA0183/include/nmea0183.h"
#include <Pacpus/kernel/InputOutputInterface.h>

#include "GpsFrames.h"

#include <QSemaphore>
#include <QThread>
#include <QByteArray>
#include <QString>

#ifdef WIN32
#   include "../driver/win32SerialPort.h"
//#   include "network/gpsServerSocket.h"
#else
#   include "../driver/PosixSerialPort.h"
#endif
//#include "DbitePlayer/SensorTcpServer.h"
#include "Pacpus/PacpusTools/ShMem.h"

// Export macro for PluginTest DLL for Windows only
#ifdef WIN32
#   ifdef GPS_EXPORTS
        // make DLL
#       define GPS_API __declspec(dllexport)
#   else
        // use DLL
#       define GPS_API __declspec(dllimport)
#   endif
#else
    // On other platforms, simply ignore this
#   define GPS_API /* nothing */
#endif

class QFile;

namespace pacpus {

/*!
* The definition of the GpsComponent class
*/
class GPS_API GpsComponent
        : public QThread
        , public ComponentBase
        , public GpsFrames
{
  Q_OBJECT

public:
  GpsComponent(QString name);
  ~GpsComponent();

  virtual void stopActivity(); /*!< to stop the processing thread */
  virtual void startActivity(); /*!< to start the processing thread */
  virtual ComponentBase::COMPONENT_CONFIGURATION configureComponent(XmlComponentConfig config);


  void enableSocketServer(int portNumber); /*!< to enable the socket server interface on port portNumber*/
  void disableSocketServer(); /*!< to disable the socket server interface */

  void setPortCOM(const char * port);



public slots:
  void unlockProcessing(int v); /*!< to unlock the processing thread */

protected:
private:
  void run(); /*!< the main loop of the thread */
  int frameType(const QByteArray& frame); /*!< to get the type of the frame */
  int decodeFrame(int type); /*!< parse the string in the NMEA0183 class to get the frame informations */
  bool currentFrameIsPps(); /*!< checks if a frame is an out-of-band pps signal */
  bool analyzeFrame(); /*!< reconstruct the entire frame starting from the received pieces */

  void addOutputs();

#ifdef WIN32
  Win32SerialPort * serialPort;
  //GpsServerSocket * serverSocket;
#else
  PosixSerialPort * serialPort;
#endif
  // a pointer to the TCP server that send data on the network
  //SensorTcpServer * tcpServer_;

  static const int MAX_FRAMES = 9;

  struct FrameTypeMap {
    char *code;
    int type;
  };

  static FrameTypeMap frameTypes[MAX_FRAMES];

  NMEA0183 nmea0183_;
  QSemaphore semaphore_;

  FRAME* currentFrame_;

  FRAME frameToDecode_;

  road_time_t lastPpsTime_;

  bool newFrameToDecode_;
  bool startOfFrame_;
  bool endOfFrame_;
  int nextByteToProcess_;


  unsigned int ppsIndex_;

  DbiteFile ppshdFile; /*!< pointer to the pps dbt file */
  DbiteFile ggahdFile; /*!< pointer to the gga dbt file */
  DbiteFile gsahdFile; /*!< pointer to the gsa dbt file */
  DbiteFile gsthdFile; /*!< pointer to the gst dbt file */
  DbiteFile gsvhdFile; /*!< pointer to the gsv dbt file */
  DbiteFile hdthdFile; /*!< pointer to the hdt dbt file */
  DbiteFile rmchdFile; /*!< pointer to the rmc dbt file */
  DbiteFile rothdFile; /*!< pointer to the rot dbt file */
  DbiteFile vtghdFile; /*!< pointer to the vtg dbt file */
  DbiteFile zdahdFile; /*!< pointer to the zda dbt file */

  DbiteFile gpsSynchroFile; /*!< pointer to the zda dbt file */

  QFile * file;

  bool ppsRecording;
  bool ggaRecording;
  bool gsaRecording;
  bool gstRecording;
  bool gsvRecording;
  bool hdtRecording;
  bool rmcRecording;
  bool rotRecording;
  bool vtgRecording;
  bool zdaRecording;

  QString portName_;

  ShMem * shMem_;

  TimestampedGgaFrame mGga;
  TimestampedVtgFrame mVtg;
  TimestampedHdtFrame mHdt;

  // Declaration of outputs
  OutputInterface<TimestampedGgaFrame, GpsComponent>* outGga_;
  OutputInterface<TimestampedVtgFrame, GpsComponent>* outVtg_;
  OutputInterface<TimestampedHdtFrame, GpsComponent>* outHdt_;

  /// For the socket server
  /* Qt3 version - 17/12/2007
  BOOL socketServerEnabled;
  void *frameToSend;
  QCustomEvent *evt;
  */

  /*
  * VC6 doesn't support out-of-class template defintions
  * see on http://support.microsoft.com kb q243451 and more particulary 241949
  */
  template<typename T>
  void sendDataToServerSocket(T /*frame*/, int /*type*/)
  {
    /* Qt3 version - 17/12/2007
    if (socketServerEnabled)
    {
      frameToSend = new T;
      *(T*)frameToSend = frame;
      evt = new QCustomEvent(QEvent::User + type);
      evt->setData(frameToSend);
      postEvent(serverSocket,evt);
    }
    */
  }
};

} // namespace pacpus

#endif // GPSCOMPONENT_H
