/// Purpose:     Implementation of the GpsComponent class
///
/// @date created 2005/09/02 - 14:16
/// @author Gerald Dherbomez
/// @copyright Heudiasyc UMR UTC/CNRS 6599
/// @version $Id: gpsComponent.cpp 1278 2013-01-16 16:40:11Z bonnetst $

#include "gpsComponent.h"

#include <qfile.h>
#include <qapplication.h>

#include "Pacpus/kernel/ComponentFactory.h"
#include "Pacpus/kernel/Log.h"
#include "Pacpus/kernel/DbiteFileTypes.h"

#include <qdebug.h>
#include <iostream>

#define PACPUS_PI 3.1415926

namespace pacpus {

/// Construct the factory
ComponentFactory<GpsComponent> sFactory("GpsComponent");

DECLARE_STATIC_LOGGER("pacpus.base.GpsComponent");

#define UNKNOWN_NMEA_FRAME -1

//////////////////////////////////////////////////////////////////////////
/*!
* Constructor of GpsComponent class
*/
//////////////////////////////////////////////////////////////////////////
GpsComponent::GpsComponent(QString name)
    : semaphore_(0)
    , ComponentBase(name)

{

  ppsRecording = true;//TRUE
  ggaRecording = true;
  gsaRecording = true;
  gstRecording = true;
  gsvRecording = true;
  hdtRecording = true;
  rmcRecording = true;
  rotRecording = true;
  vtgRecording = true;
  zdaRecording = true;

/*  ppshdFile = NULL;
  ggahdFile = NULL;
  gsahdFile = NULL;
  gsthdFile = NULL;
  gsvhdFile = NULL;
  hdthdFile = NULL;
  rmchdFile = NULL;
  rothdFile = NULL;
  vtghdFile = NULL;
  zdahdFile = NULL;
*/


  nextByteToProcess_ = 0;
  currentFrame_ = NULL;
  newFrameToDecode_ = false;
  startOfFrame_ = false;
  endOfFrame_ = false;
}


/*!
* Destructor of the GpsComponent class
*/
GpsComponent::~GpsComponent()
{

}


void GpsComponent::addOutputs()
{
  addOutput<TimestampedGgaFrame, GpsComponent>("ggaOut");
  addOutput<TimestampedVtgFrame, GpsComponent>("vtgOut");
  addOutput<TimestampedHdtFrame, GpsComponent>("hdtOut");
}



ComponentBase::COMPONENT_CONFIGURATION GpsComponent::configureComponent(XmlComponentConfig config)
{
  setPortCOM( config.getProperty("port").toLatin1() );
  setRecording ((config.getProperty("recording") == "true" ? true : false));

  return ComponentBase::CONFIGURED_OK;
}


void GpsComponent::setPortCOM(const char * port)
{
  portName_ = port;
}


void GpsComponent::enableSocketServer(int /*portNumber*/)
{

}


void GpsComponent::disableSocketServer()
{

}


/************************************************************************/
/* Main loop of the thread                                              */
/************************************************************************/
void GpsComponent::run()
{
  int type = -1;

  // parity:0-4=no,odd,even,mark,space
  // byteSize:number of bits/byte, 4-8
  // baudRate:port speed (ex:38400)
  // stopBits:0,1,2 = 1, 1.5, 2

  serialPort->configurePort(xmlParameters().getProperty("baudrate").toLong(),
    xmlParameters().getProperty("bytesize").toUInt(),
    //  config.getProperty("parity").at(0).toLatin1(),
    xmlParameters().getProperty("parity").toShort(),
    xmlParameters().getProperty("stopbits").toUInt() - 1);


  serialPort->start();

  while (isActive())
  {
    // If there are no bytes remaining in the current frame, delete it
    if ((currentFrame_ != NULL) && (nextByteToProcess_ >= currentFrame_->data.length())) {
      delete currentFrame_;
      currentFrame_ = NULL;
    }

    // If there is no current frame, get one. Wait for one if none is available.
    if (currentFrame_ == NULL) {
      semaphore_.acquire();
      if  ((currentFrame_ = serialPort->getNextFrame()) == NULL) {
        // Should never get to this point as the semaphore counts the frames, unless
        // the thread is being stopped.
        continue;
      } else {
        nextByteToProcess_ = 0;
      }
    }

    // Check if we got a PPS frame
    if (currentFrameIsPps()) {
      lastPpsTime_ = currentFrame_->t;
      decodeFrame(SIGNAL_PPS);
    } else if (analyzeFrame()) {
      // a new complete NMEA frame has arrived, decode it.
      type = frameType(frameToDecode_.data);
      if (type != -1) {
        if (decodeFrame(type) == -1) {
            qWarning("Failed to decode the dataframe\n");
        }
      }

      setState(ComponentBase::MONITOR_OK);
    }
  }

  qDebug() << "The thread of " << name() << " GPS component is stopped";
}



/************************************************************************/
/* Method to stop the thread                                            */
/************************************************************************/
void GpsComponent::stopActivity()
{
  setActive( false );
  unlockProcessing(1); // to unblock the thread that is waiting new data

  serialPort->THREAD_ALIVE = false;

  if (!serialPort->wait(2000))
  {
    serialPort->terminate();
    qDebug("The Win32SerialPort thread blocks anormally, it has been killed !!");
  }
  if ( !serialPort->closePort() )
    qDebug("Failed to close the port");
  else
    qDebug("The port is closed");
  delete serialPort;

  if (ppshdFile.isOpen())
	  ppshdFile.close();
  if (ggahdFile.isOpen())
	  ggahdFile.close();
  if (gsahdFile.isOpen())
    gsahdFile.close();
  if (gsthdFile.isOpen())
    gsthdFile.close();
  if (gsvhdFile.isOpen())
    gsvhdFile.close();
  if (hdthdFile.isOpen())
    hdthdFile.close();
  if (rmchdFile.isOpen())
	  rmchdFile.close();
  if (rothdFile.isOpen())
	  rothdFile.close();
  if (vtghdFile.isOpen())
	  vtghdFile.close();
  if (zdahdFile.isOpen())
	  zdahdFile.close();
}


/************************************************************************/
/* Method to start the thread                                            */
/************************************************************************/
void GpsComponent::startActivity()
{
  ppsIndex_ = 0;

  outGga_ = getTypedOutput<TimestampedGgaFrame, GpsComponent>("ggaOut");
  outVtg_ = getTypedOutput<TimestampedVtgFrame, GpsComponent>("vtgOut");
  outHdt_ = getTypedOutput<TimestampedHdtFrame, GpsComponent>("hdtOut");


  setActive( true );

#if WIN32
  serialPort = new Win32SerialPort(portName_.toLatin1());
  // Asynchrone
  serialPort->setMode(FILE_FLAG_OVERLAPPED);
  // Synchrone
  //serialPort->setMode(0);
#else
  serialPort = new PosixSerialPort(portName_.toLatin1());
#endif

  if (!serialPort->openPort(portName_.toLatin1()))
  {
    qDebug() << "Failed to open the port " << portName_;
    qDebug() << "The GPS Component " << name() << " didn't start";
    return;
  }

  serialPort->THREAD_ALIVE = true;
  if (!QApplication::connect(serialPort,SIGNAL(newDataAvailable(int)),this, SLOT(unlockProcessing(int))))
    qWarning("Failed to connect SIGNAL(newDataAvailable(int)) with SLOT(unlockProcessing(int)\n" );
  start();
  qDebug() << "The Component " << name() << " is started";
}


/************************************************************************/
/* To get the type of the received frame                                */
/************************************************************************/

// The NMEA frame code <-> pacpus type map
GpsComponent::FrameTypeMap GpsComponent::frameTypes[GpsComponent::MAX_FRAMES] = {
  { "PPS", SIGNAL_PPS },
  { "$GPGGA", TRAME_GGA_DBL},
  { "$GPGSA", TRAME_GSA },
  { "$GPHDT", TRAME_HDT },
  { "$GPGST", TRAME_GST },
  { "$GPGSV", TRAME_GSV },
  { "$GPRMC", TRAME_RMC },
  { "$GPVTG", TRAME_VTG },
  { "$GPZDA", TRAME_ZDA },
};

int GpsComponent::frameType(const QByteArray& frame)
{
  for (int i = 0; i < MAX_FRAMES; i++) {
    if (frame.startsWith(frameTypes[i].code)) {
      return frameTypes[i].type;
    }
  }

  return UNKNOWN_NMEA_FRAME;
}

/************************************************************************/
/* Returns true is the frame is a pps, without updating frameToDecode_  */
/* which may already contain an in-band NMEA partial frame              */
/************************************************************************/
bool GpsComponent::currentFrameIsPps() {
  if (currentFrame_->data == "PPS") {
    nextByteToProcess_ = 3;
    return true;
  } else {
    return false;
  }
}

/************************************************************************/
/* Reconstruct the frame starting from the received fragments           */
/************************************************************************/
bool GpsComponent::analyzeFrame()
{
  // Process the remaining bytes in the current frame
  while(nextByteToProcess_ < currentFrame_->data.length()) {

    char currentChar = currentFrame_->data[nextByteToProcess_++];

    // first looking for start-of-frame
    if (!startOfFrame_ && (currentChar == '$')) {
      startOfFrame_ = true;
      endOfFrame_ = false;

      frameToDecode_.t = currentFrame_->t;
      frameToDecode_.data.clear();
    } else if (startOfFrame_ && !endOfFrame_ && (currentChar == '\n')) {
      // Looking for end-of-frame
      startOfFrame_ = false;
      endOfFrame_ = true;
      frameToDecode_.t = road_time();
      //frameToDecode_.tr = static_cast<road_timerange_t>(currentFrame_->t - frameToDecode_.t);
      return true;  // There is a new frame to decode
    }

    if ((startOfFrame_) && (!endOfFrame_)) {
      frameToDecode_.data.append(currentChar);
    }
  }
  return false; // No new frame to decode, wait for more data
}


void GpsComponent::unlockProcessing(int v)
{
  // new frame available
  semaphore_.release( v );
}


int GpsComponent::decodeFrame(int type)
{
  double lat_rad = 0, lon_rad = 0;
  int indexGSV = 0;
  int indexGSA = 0;


  SENTENCE sentence;
  sentence.Sentence = frameToDecode_.data;

  switch(type)
  {
  case UNKNOWN_NMEA_FRAME:
    qDebug("Unknown frame received !");
    break;
  case SIGNAL_PPS:
    ppsFrame.setRoadTime(lastPpsTime_);
    *ppsFrame.getFrameData() = ppsIndex_++;

    if ( (isRecording())  && (ppsRecording) )
    {
		if (!(ppshdFile.isOpen()))
        ppshdFile.open((char *)(name() + "_pps.dbt").toLatin1().data(),WriteMode,TYPE_INT,sizeof(int));
    if ( ppshdFile.writeRecord(frameToDecode_.t,frameToDecode_.tr,(char *) ppsFrame.getFrameData(),sizeof(int)) == 0)
        qWarning("Failed to record PPS data ...\n");
    }
    ppsFrame.notifyObservers();

    break;

  case TRAME_GGA_DBL:
    if (!nmea0183_.Gga.Parse(sentence)) {
      LOG_ERROR("Failed to parse the frame " << nmea0183_.Gga.ErrorMessage.toLatin1().data());
    } else {
      lat_rad = nmea0183_.Gga.Position.Latitude.GetDecimalDegrees()*PACPUS_PI/180;
      lon_rad = nmea0183_.Gga.Position.Longitude.GetDecimalDegrees()*PACPUS_PI/180;
      ggaFrame.getFrameData()->H = nmea0183_.Gga.Time.time().hour();
      ggaFrame.getFrameData()->Mi = nmea0183_.Gga.Time.time().minute();
      ggaFrame.getFrameData()->S = nmea0183_.Gga.Time.time().second();
      ggaFrame.getFrameData()->Ms = nmea0183_.Gga.Time.time().msec();
      ggaFrame.getFrameData()->lon = lon_rad;
      ggaFrame.getFrameData()->lat = lat_rad;
      ggaFrame.getFrameData()->ind_qualite = nmea0183_.Gga.GPSQuality;
      ggaFrame.getFrameData()->nb_sat = nmea0183_.Gga.NumberOfSatellitesInUse;
      ggaFrame.getFrameData()->hdop = nmea0183_.Gga.HorizontalDilutionOfPrecision;
      ggaFrame.getFrameData()->alt_msl = nmea0183_.Gga.AntennaAltitudeMeters;
      ggaFrame.getFrameData()->d_geoidal = nmea0183_.Gga.GeoidalSeparationMeters;
      ggaFrame.getFrameData()->age = nmea0183_.Gga.AgeOfDifferentialGPSDataSeconds;
      ggaFrame.getFrameData()->dir_lat = ( (nmea0183_.Gga.Position.Latitude.Northing == North) ? 'N' : 'S' );
      ggaFrame.getFrameData()->dir_lon = ( (nmea0183_.Gga.Position.Longitude.Easting == East) ? 'E' : 'W' );
      ggaFrame.getFrameData()->ref_station_ID = nmea0183_.Gga.DifferentialReferenceStationID;
      ggaFrame.setRoadTime(frameToDecode_.t);

      sendDataToServerSocket(*ggaFrame.getFrameData(),type);
      ggaFrame.notifyObservers();

      memcpy(&mGga.frame, ggaFrame.getFrameData(), sizeof(trame_gga_dbl));
      mGga.time = ggaFrame.getRoadTime();
      mGga.timerange = 0;

      // send GGA data to output
      checkedSend(outGga_, mGga);

      if ( (isRecording()) && (ggaRecording) ) {
		    if (!(ggahdFile.isOpen()))
	        ggahdFile.open((char *)(name() + "_gga.dbt").toLatin1().data(),WriteMode,TRAME_GGA_DBL,sizeof(trame_gga_dbl));
		    if ( !ggahdFile.writeRecord(frameToDecode_.t,frameToDecode_.tr,(char *) ggaFrame.getFrameData(),sizeof(trame_gga_dbl)))
          qWarning("Failed to record GGA data ...\n");
      }
      //printf("lon=%f lat=%f\n",ggaFrame.getFrameData()->lon, ggaFrame.getFrameData()->lat);
    }
    break;

  case TRAME_GSA:
    if (!nmea0183_.Gsa.Parse(sentence))
      qWarning("Failed to parse the frame %s\n", nmea0183_.Gsa.ErrorMessage.toLatin1().data());
    gsaFrame.getFrameData()->mode_select = ((nmea0183_.Gsa.OperatingMode == GSA::Manual) ? 'M' : 'A');
    gsaFrame.getFrameData()->mode_result = 0;
    if (nmea0183_.Gsa.FixMode == GSA::FixUnavailable)
      gsaFrame.getFrameData()->mode_result = 1;
    if (nmea0183_.Gsa.FixMode == GSA::TwoDimensional)
      gsaFrame.getFrameData()->mode_result = 2;
    if (nmea0183_.Gsa.FixMode == GSA::ThreeDimensional)
      gsaFrame.getFrameData()->mode_result = 3;
    for (indexGSA = 0 ; indexGSA<12 ; indexGSA++)
      gsaFrame.getFrameData()->SV_PRN[indexGSA] = nmea0183_.Gsa.SatelliteNumber[indexGSA];
    gsaFrame.getFrameData()->pdop = nmea0183_.Gsa.PDOP;
    gsaFrame.getFrameData()->hdop = nmea0183_.Gsa.HDOP;
    gsaFrame.getFrameData()->vdop = nmea0183_.Gsa.VDOP;
    gsaFrame.setRoadTime(frameToDecode_.t);

    gsaFrame.notifyObservers();

    sendDataToServerSocket(*gsaFrame.getFrameData(),type);

    if ( (isRecording()) && (gsaRecording) )
    {
		if (!(gsahdFile.isOpen()))
		  gsahdFile.open((char *)(name() + "_gsa.dbt").toLatin1().data(),WriteMode, TRAME_GSA,sizeof(trame_gsa));
		if ( gsahdFile.writeRecord(frameToDecode_.t, frameToDecode_.tr,(char *) gsaFrame.getFrameData(),sizeof(trame_gsa)) != 1)
        qWarning("Failed to record GSA data ...\n");
    }
    break;



  case TRAME_GST:
    if (!nmea0183_.Gst.Parse( sentence ))
      qWarning("Failed to parse the frame %s\n",nmea0183_.Gst.ErrorMessage.toLatin1().data());
    gstFrame.getFrameData()->rms = nmea0183_.Gst.RMSvalue;
    gstFrame.getFrameData()->a = nmea0183_.Gst.ErrorEllipseMajor;
    gstFrame.getFrameData()->b = nmea0183_.Gst.ErrorEllipseMinor;
    gstFrame.getFrameData()->phi = nmea0183_.Gst.ErrorEllipseOrientation;
    gstFrame.getFrameData()->sigma_lat = nmea0183_.Gst.LatitudeError;
    gstFrame.getFrameData()->sigma_lon = nmea0183_.Gst.LongitudeError;
    gstFrame.getFrameData()->sigma_alt = nmea0183_.Gst.HeightError;
    gstFrame.getFrameData()->H = nmea0183_.Gst.Time.time().hour();
    gstFrame.getFrameData()->Mi = nmea0183_.Gst.Time.time().minute();
    gstFrame.getFrameData()->S = nmea0183_.Gst.Time.time().second();
    gstFrame.getFrameData()->Ms = nmea0183_.Gst.Time.time().msec();
    gstFrame.setRoadTime(frameToDecode_.t);

    sendDataToServerSocket(*gstFrame.getFrameData(),type);
    gstFrame.notifyObservers();


    if ( (isRecording()) && (gstRecording) ) {
		  if (!gsthdFile.isOpen())
			  gsthdFile.open((char *)(name() + "_gst.dbt").toLatin1().data(),WriteMode,TRAME_GST,sizeof(trame_gst));
      if ( !gsthdFile.writeRecord(frameToDecode_.t, frameToDecode_.tr,(char *) gstFrame.getFrameData(),sizeof(trame_gst)) )
        qWarning("Failed to record GST data ...\n");
    }
    break;

  case TRAME_GSV:
    indexGSV = 0;
    if (!nmea0183_.Gsv.Parse( sentence )) {
      qWarning("Failed to parse the frame %s\n",nmea0183_.Gsv.ErrorMessage.toLatin1().data());
      break;
    }
    // it's a new frame, reset stored value in case of the number of satellites
    // in view has decreased
    if (nmea0183_.Gsv.message_number == 1)
    {
      while (indexGSV < 36)
      {
        gsvFrame.getFrameData()->SatellitesInView[ indexGSV ][ 0 ] = 0;
        gsvFrame.getFrameData()->SatellitesInView[ indexGSV ][ 1 ] = 0;
        gsvFrame.getFrameData()->SatellitesInView[ indexGSV ][ 2 ] = 0;
        gsvFrame.getFrameData()->SatellitesInView[ indexGSV ][ 3 ] = 0;
        indexGSV++;
      }
    }
    gsvFrame.getFrameData()->NumberOfSatellites = nmea0183_.Gsv.NumberOfSatellites;
    gsvFrame.getFrameData()->Totalmessages = nmea0183_.Gsv.Totalmessages;

    for  ( indexGSV=4*(nmea0183_.Gsv.message_number-1); indexGSV<=(4*nmea0183_.Gsv.message_number)-1; indexGSV++ )
    {
      gsvFrame.getFrameData()->SatellitesInView[ indexGSV ][ 0 ] = nmea0183_.Gsv.SatellitesInView[ indexGSV ].SatelliteNumber;
      gsvFrame.getFrameData()->SatellitesInView[ indexGSV ][ 1 ] = nmea0183_.Gsv.SatellitesInView[ indexGSV ].ElevationDegrees;
      gsvFrame.getFrameData()->SatellitesInView[ indexGSV ][ 2 ] = nmea0183_.Gsv.SatellitesInView[ indexGSV ].AzimuthDegreesTrue;
      gsvFrame.getFrameData()->SatellitesInView[ indexGSV ][ 3 ] = nmea0183_.Gsv.SatellitesInView[ indexGSV ].SignalToNoiseRatio;
    }

    if (nmea0183_.Gsv.Totalmessages == nmea0183_.Gsv.message_number) {
      sendDataToServerSocket(*gsvFrame.getFrameData(),type);
      gsvFrame.setRoadTime(frameToDecode_.t);
      gsvFrame.notifyObservers();
    }
    if ( (isRecording()) && (gsvRecording) && (nmea0183_.Gsv.Totalmessages == nmea0183_.Gsv.message_number) )
    {
		if (!gsvhdFile.isOpen())
        gsvhdFile.open((char *)(name() + "_gsv.dbt").toLatin1().data(),WriteMode,TRAME_GSV,sizeof(trame_gsv));
		if ( gsvhdFile.writeRecord(frameToDecode_.t, frameToDecode_.tr,(char *) gsvFrame.getFrameData(),sizeof(trame_gsv)) == 0)
        qWarning("Failed to record GSV data ...\n");
    }
    break;

    case TRAME_HDT:
      if (!nmea0183_.Hdt.Parse( sentence ))
        qWarning("Failed to parse the frame %s\n",nmea0183_.Hdt.ErrorMessage.toLatin1().data());
      hdtFrame.getFrameData()->DegreesTrue = nmea0183_.Hdt.DegreesTrue;
      hdtFrame.setRoadTime(frameToDecode_.t);

      sendDataToServerSocket(hdtFrame,type);
      hdtFrame.notifyObservers();

      memcpy(&mHdt.frame, hdtFrame.getFrameData(), sizeof(trame_hdt));
      mHdt.time = hdtFrame.getRoadTime();
      mHdt.timerange = 0;

      // send HDT data to output
      checkedSend(outHdt_, mHdt);

      if ( (isRecording()) && (hdtRecording) )
      {
		  if (!hdthdFile.isOpen())
			  hdthdFile.open((char *)(name() + "_hdt.dbt").toLatin1().data(),WriteMode,TRAME_HDT,sizeof(trame_hdt));
      if ( hdthdFile.writeRecord(frameToDecode_.t, frameToDecode_.tr,(char *) hdtFrame.getFrameData(),sizeof(trame_hdt)) == 0)
          qWarning("Failed to record HDT data ...\n");
      }
      break;

    case TRAME_RMC:
      if (!nmea0183_.Rmc.Parse( sentence ))
        qWarning("Failed to parse the frame %s\n",nmea0183_.Rmc.ErrorMessage.toLatin1().data());
      rmcFrame.getFrameData()->H = nmea0183_.Rmc.Time.time().hour();
      rmcFrame.getFrameData()->Mi = nmea0183_.Rmc.Time.time().minute();
      rmcFrame.getFrameData()->S = nmea0183_.Rmc.Time.time().second();
      rmcFrame.getFrameData()->Ms = nmea0183_.Rmc.Time.time().msec();
      rmcFrame.getFrameData()->AA = nmea0183_.Rmc.Time.date().year();
      rmcFrame.getFrameData()->MM = nmea0183_.Rmc.Time.date().month();
      rmcFrame.getFrameData()->JJ = nmea0183_.Rmc.Time.date().day();
      rmcFrame.getFrameData()->lat = nmea0183_.Rmc.Position.Latitude.GetDecimalDegrees()*PACPUS_PI/180;
      rmcFrame.getFrameData()->dir_lat = ( (nmea0183_.Rmc.Position.Latitude.Northing == North) ? 'N' : 'S');
      rmcFrame.getFrameData()->lon = nmea0183_.Rmc.Position.Longitude.GetDecimalDegrees()*PACPUS_PI/180;;
      rmcFrame.getFrameData()->dir_lon = ( (nmea0183_.Rmc.Position.Longitude.Easting == East) ? 'E' : 'W' );
      rmcFrame.getFrameData()->magnet_var = nmea0183_.Rmc.MagneticVariation;
      rmcFrame.getFrameData()->dir_magnet_var = ( (nmea0183_.Rmc.MagneticVariationDirection == East) ? 'E' : 'W');
      rmcFrame.getFrameData()->mode = -1;
      if (nmea0183_.Rmc.ModeIndication == "A")
        rmcFrame.getFrameData()->mode = 1;
      if (nmea0183_.Rmc.ModeIndication == "D")
        rmcFrame.getFrameData()->mode = 2;
      if (nmea0183_.Rmc.ModeIndication == "N")
        rmcFrame.getFrameData()->mode = 0;
      rmcFrame.getFrameData()->track_true_north = nmea0183_.Rmc.TrackMadeGoodDegreesTrue;
      rmcFrame.getFrameData()->valid_data = ( (nmea0183_.Rmc.IsDataValid == True) ? 1 : 0 );
      rmcFrame.getFrameData()->vitesse = nmea0183_.Rmc.SpeedOverGroundKnots * 1852.0 / 3600.0; // 1 knot = 1852 m/h
      rmcFrame.setRoadTime(frameToDecode_.t);

      sendDataToServerSocket(*rmcFrame.getFrameData(),type);
      rmcFrame.notifyObservers();

      if ( (isRecording()) && (rmcRecording) )
      {
        if (!rmchdFile.isOpen())
          rmchdFile.open((char *)(name() + "_rmc.dbt").toLatin1().data(),WriteMode,TRAME_RMC,sizeof(trame_rmc));
        if (rmchdFile.writeRecord(frameToDecode_.t ,frameToDecode_.tr,(char *) rmcFrame.getFrameData(),sizeof(trame_rmc)) == 0)
          qWarning("Failed to record RMC data ...\n");
      }
      break;

    case TRAME_ROT:
      if (!nmea0183_.Rot.Parse( sentence ))
        qWarning("Failed to parse the frame %s\n",nmea0183_.Rot.ErrorMessage.toLatin1().data());
      rotFrame.getFrameData()->RateOfTurn = nmea0183_.Rot.RateOfTurn;
      rotFrame.getFrameData()->valid_data = ( (nmea0183_.Rot.IsDataValid == True) ? 1 : 0 );
      rotFrame.setRoadTime(frameToDecode_.t);

      sendDataToServerSocket(*rotFrame.getFrameData(),type);
      rotFrame.notifyObservers();

      if ( (isRecording()) && (rotRecording) )
      {
        if (!rothdFile.isOpen())
          rothdFile.open((char *)(name() + "_rot.dbt").toLatin1().data(),WriteMode,TRAME_ROT,sizeof(trame_rot));
        if ( rothdFile.writeRecord(frameToDecode_.t ,frameToDecode_.tr,(char *) rotFrame.getFrameData(),sizeof(trame_rot)) == 0)
          qWarning("Failed to record ROT data ...\n");
      }
      break;

    case TRAME_VTG:
      if (!nmea0183_.Vtg.Parse( sentence )) {
        LOG_WARN("Failed to parse the frame " << nmea0183_.Vtg.ErrorMessage);
      } else {
        vtgFrame.getFrameData()->v = nmea0183_.Vtg.SpeedKilometersPerHour;
        vtgFrame.getFrameData()->track_true_north = nmea0183_.Vtg.TrackDegreesTrue;
        vtgFrame.getFrameData()->track_magnet_north = nmea0183_.Vtg.TrackDegreesMagnetic;
        vtgFrame.setRoadTime(frameToDecode_.t);

        sendDataToServerSocket(vtgFrame,type);
        vtgFrame.notifyObservers();

        memcpy(&mVtg.frame, vtgFrame.getFrameData(), sizeof(trame_vtg));
        mVtg.time = vtgFrame.getRoadTime();
        mVtg.timerange = 0;

        // send VTG data to output
        checkedSend(outVtg_, mVtg);

        if (isRecording() && vtgRecording) {
          if (!vtghdFile.isOpen())
            vtghdFile.open((char *)(name() + "_vtg.dbt").toLatin1().data(),WriteMode,TRAME_VTG,sizeof(trame_vtg));
		      if ( vtghdFile.writeRecord(frameToDecode_.t, frameToDecode_.tr,(char *) vtgFrame.getFrameData(),sizeof(trame_vtg)) == 0)
            qWarning("Failed to record VTG data ...\n");
        }
      }
      break;

    case TRAME_ZDA:
      if (!nmea0183_.Zda.Parse( sentence )) {
        LOG_WARN("Failed to parse the frame " << nmea0183_.Zda.ErrorMessage);
      }
      zdaFrame.getFrameData()->H = nmea0183_.Zda.Time.time().hour();
      zdaFrame.getFrameData()->Mi = nmea0183_.Zda.Time.time().minute();
      zdaFrame.getFrameData()->S = nmea0183_.Zda.Time.time().second();
      zdaFrame.getFrameData()->Ms = nmea0183_.Zda.Time.time().msec();
      zdaFrame.getFrameData()->AA = nmea0183_.Zda.Time.date().year();
      zdaFrame.getFrameData()->MM = nmea0183_.Zda.Time.date().month();
      zdaFrame.getFrameData()->JJ = nmea0183_.Zda.Time.date().day();
      zdaFrame.getFrameData()->H_offset = nmea0183_.Zda.LocalHourDeviation;
      zdaFrame.getFrameData()->Mi_offset = nmea0183_.Zda.LocalMinutesDeviation;

      sendDataToServerSocket(*zdaFrame.getFrameData(), type);
      zdaFrame.setRoadTime(frameToDecode_.t);
      zdaFrame.notifyObservers();

      if ( (isRecording()) && (zdaRecording) ) {
		  if (!zdahdFile.isOpen())
          zdahdFile.open((char *)(name() + "_zda.dbt").toLatin1().data(),WriteMode,TRAME_ZDA,sizeof(trame_zda));
      if ( zdahdFile.writeRecord(frameToDecode_.t, frameToDecode_.tr,(char *) zdaFrame.getFrameData(),sizeof(trame_zda)) == 0) {
            LOG_WARN("Failed to record ZDA data ...");
          }
      }

      /*
      if (isRecording() && zdaRecording && ppsRecording) {
          memcpy(&(synchroFrame.getFrameData()->zda), zdaFrame.getFrameData(), sizeof(zdaFrame));
          synchroFrame.getFrameData()->timeOffset = frameToDecode_.t - lastPpsTime_;
          synchroFrame.getFrameData()->ppsTime = lastPpsTime_;
          synchroFrame.getFrameData()->zdaTime = frameToDecode_.t;
          synchroFrame.setRoadTime(frameToDecode_.t);

          if (!gpsSynchroFile.isOpen()) {
            gpsSynchroFile.open((name() + "_synchro.dbt").toStdString(), WriteMode, GPS_SYNCHRO_FRAME, sizeof(GpsSynchroFrame) );
          }

          if (gpsSynchroFile.writeRecord(frameToDecode_.t, frameToDecode_.t - lastPpsTime_, reinterpret_cast<const char *>(synchroFrame.getFrameData()), sizeof(GpsSynchroFrame)) == 0) {
            LOG_WARN("Failed to record GPS SYNCHRO data ...");
          }
      }
      */

      break;

    default:
      return 0;
  }

  return 1;
}

} // namespace pacpus
