/**
 *
 * Distributed under the UTC Heudiascy Pacpus License, Version 1.0.
 * Copyright (c) UTC Heudiasyc 2010 - 2013. All rights reserved.
 *
 * See the LICENSE file for more information or a copy at:
 *   http://www.hds.utc.fr/~kurdejma/LICENSE_1_0.txt
 *
 */

#ifndef COMPONENTMANAGER_H
#define COMPONENTMANAGER_H

#include <cstddef>

#include <QMap>
#include <QList>
#include <QPluginLoader>

#include "pacpus.h"
#include "ComponentFactoryBase.h"
#include "PacpusPluginInterface.h"
#include "XmlConfigFile.h"

namespace pacpus {

class ComponentBase;

typedef QMap<QString, ComponentBase *> ComponentMap;
typedef QMap<QString, ComponentFactoryBase *> FactoryMap;

/** ComponentManager
 * @brief Singleton recording the components and managing them.
 */
class ComponentManager
{
    friend class ComponentBase;

public:
    /// @returns a pointer to the ComponentManager object
    /// @deprecated use getInstance()
    PACPUS_DEPRECATED_MSG( static PACPUSLIB_API ComponentManager * create(), "use 'getInstance()'" );

    /** Get the singleton of the ComponentManager class.
     * @return Pointer to the ComponentManager singleton.
     */
    static PACPUSLIB_API ComponentManager* getInstance();

    /** Destroy the ComponentManager singleton.
     * After this call, every pointer to the ComponentManager becomes invalid.
     */
    static PACPUSLIB_API void destroy();

    struct destroyer {
        void operator()(ComponentManager * mgr) const
        {
            if (!mgr) {
                return;
            }
            mgr->destroy();
        }
    };

    /** Load the components included in the XML config file.
     * @param file Name of the XML file.
     * @return Number of components loaded by the manager.
     */
    std::size_t PACPUSLIB_API loadComponents(QString file);

    /** Start all the components
     * @return True if all the component has been started, otherwise false.
     */
    int PACPUSLIB_API start();

    /** Start only the component passed in parameter.
     * @param component Component to start.
     * @return True if the component exists and has been started, otherwise false.
     */
    int PACPUSLIB_API start(QString component);

    /** Stop all the components
     * @return True if all the component has been stopped, otherwise false.
     */
    int PACPUSLIB_API stop();

    /** Stop only the component passed in parameter.
     * @param component Component to stop.
     * @return True if the component has been stopped, otherwise false.
     */
    int PACPUSLIB_API stop(QString component);

    /** Get a pointer to the component referred by @em name.
     * @param name Name of the component.
     * @return Pointer to the component if it exists, otherwise @em NULL.
     */
    PACPUSLIB_API ComponentBase* getComponent(QString name);

    /** Get the list of all the names of the component known by the manager.
     * @return List of all the component's name.
     */
    PACPUSLIB_API QStringList getAllComponentsName();

    /** Load a new plugin from the file filename (it may be a .so/.dll file)
     * @param filename Name of the shared object or the dll.
     * @return True if the plugin has been loaded, otherwise false.
     */
    bool PACPUSLIB_API loadPlugin(QString filename);

private:
    /// Create a new component of type 'type' and with the name 'name'
    bool createComponent(QString type, QString name);

    bool registerComponent(ComponentBase * addr, QString name);
    bool registerComponentFactory(ComponentFactoryBase * addr, QString type);

    bool unRegisterComponent(QString name);
    bool unRegisterComponentFactory(QString type);

    // Allow 2 functions to access to private members of ComponentManager
    friend void ComponentFactoryBase::addFactory(ComponentFactoryBase * addr, const QString & type);
    friend void ComponentFactoryBase::addComponent(const QString & name);

    /// private constructor accessible only via static create() function
    ComponentManager();

    /// private destructor accessible only via static destroy() function
    ~ComponentManager();

private:
    /// The static pointer to this object (ComponentManager is a singleton)
    static ComponentManager * mInstance;

    /// The map of available factories of component
    FactoryMap factoryMap_;

    /// The map of loaded components
    ComponentMap componentMap_;

    /// a pointer on the xml interface
    pacpus::XmlConfigFile * xmlTree_;

    /// a list of QObject in which plugins are embedded
    QObjectList pluginList_;

    /// The object used to load the plugins
    QPluginLoader pluginLoader_;
};

} // namespace pacpus

#endif // COMPONENTMANAGER_H
