#ifndef FILE_H
#define FILE_H

/* 3dv-client/file.h
 *
 * Copyright (C) 2013 VisLab
 *
 * This file is part of 3dv-client; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */  

#include <lib3dv/image.h>
#include <lib3dv/motion.h>
#include <lib3dv/classification.h>
#include <lib3dv/obstacle.h>
#include <lib3dv/point.h>
#include <lib3dv/terrain.h>
#include <lib3dv/calibration.h>


#include <boost/archive/binary_oarchive.hpp>
#include <boost/archive/text_oarchive.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/filesystem.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/serialization/serialization.hpp>
#include <boost/serialization/vector.hpp>
#include <boost/date_time/posix_time/time_serialize.hpp>

#include <stdint.h>
#include <vector>

#include "pnm.h"

struct device_params
{
    lib3dv::calibration::camera_intrinsics intrinsics;
    lib3dv::calibration::position position;
    lib3dv::calibration::orientation orientation;
    float baseline;
    uint32_t downsample;
    double area_step;
};


namespace boost
{
    namespace serialization
    {
        template<class Archive>
        void serialize(Archive& archive, lib3dv::point3& point, const unsigned int version)
        {
            archive & point.m_x;
            archive & point.m_y;
            archive & point.m_z;
        }
        
        template<class Archive>
        void serialize(Archive& archive, lib3dv::terrain& terrain, const unsigned int version)
        {
            archive & terrain.m_data;
        }

        template<class Archive>
        void serialize(Archive& archive, lib3dv::stixel& stixel, const unsigned int version)
        {
            archive & stixel.m_dx;
            archive & stixel.m_dy;
            archive & stixel.m_x;
            archive & stixel.m_y;
            archive & stixel.m_z;
            archive & stixel.m_height;       
        }

        template<class Archive>
        void serialize(Archive& archive, lib3dv::obstacle& obstacle, const unsigned int version)
        {
            archive & obstacle.m_guid;
            archive & obstacle.m_data;
        }
  
        template<class Archive>
        void serialize(Archive& archive, lib3dv::pose& pose, const unsigned int version)
        {
            archive & pose.m_timestamp;
            archive & pose.m_type;
            archive & pose.m_data;
        }

        template<class Archive>
        void serialize(Archive& archive, lib3dv::motion& motion, const unsigned int version)
        {
           archive & motion.m_poses;
        }
        
        template<class Archive>
        void serialize(Archive& archive, lib3dv::candidate& candidate, const unsigned int version)
        {
                archive &  candidate.m_guid;
                archive &  candidate.m_x0;
                archive &  candidate.m_y0;
                archive &  candidate.m_x1;
                archive &  candidate.m_y1;
                archive &  candidate.m_confidence;
                archive &  candidate.m_wp_lb;
                archive &  candidate.m_category;
        }

        template<class Archive>
        void serialize(Archive& archive, lib3dv::classification& classification, const unsigned int version)
        {
                archive & classification.m_timestamp;
                archive & classification.m_candidates;
        }
        
    }
}


namespace data_file_format
{
    enum types {RAW, BINARY, TEXT, GNUPLOT};

    inline std::istream& operator>>(std::istream& in, types& format_type)
    {
        std::string token;
        
        in >> token;

        if(token == "text")
            format_type = TEXT;
        else if(token == "binary")
            format_type = BINARY;
        else if(token == "gnuplot")
            format_type = GNUPLOT;
        else
            format_type = RAW;

        return in;
    }
    
    inline std::ostream& operator<<(std::ostream& out, types& format_type)
    {
        out << (format_type == TEXT ? "text" : (format_type == BINARY ? "binary" : (format_type == GNUPLOT ? "gnuplot" : "raw")));

        return out;
    }
}

#ifdef ARCHIVE

namespace archive_file_format
{
    enum types { ZIP, TAR };
    
    inline std::istream& operator>>(std::istream& in, types& archive_type)
    {
        std::string token;

        in >> token;

        if(token == "zip")
            archive_type = ZIP;
        else if(token == "tar")
            archive_type = TAR;

        return in;
    }
    
    inline std::ostream& operator<<(std::ostream& out, types& archive_type)
    {
        out << (archive_type == ZIP ? "zip" : "tar");
        
        return out;
    }
}

struct archive;

archive* open_archive(const boost::filesystem::path& path, const std::string& archive_name, archive_file_format::types format,device_params params, uint8_t log_level);

bool close_archive(archive* archive, uint8_t log_level);


bool write_image_archive(boost::shared_ptr<const lib3dv::image> image, uint32_t guid, uint8_t guid_type, archive* archive, bool autonumber, uint8_t log_level);

bool write_terrain_archive(boost::shared_ptr<const lib3dv::terrain> terrain, uint32_t guid, uint8_t guid_type, archive* archive, bool autonumber, data_file_format::types format, uint8_t log_level);

bool write_obstacles_archive(boost::shared_ptr<const std::vector<lib3dv::obstacle> > obstacles, uint32_t guid, uint8_t guid_type, archive* archive, bool autonumber, data_file_format::types format, uint8_t log_level);

bool write_motion_archive(boost::shared_ptr<const lib3dv::motion> motion, uint32_t guid, uint8_t guid_type, archive* archive, bool autonumber, data_file_format::types format, uint8_t log_level);

bool write_classification_archive(boost::shared_ptr<const lib3dv::classification> classificat, uint32_t guid, uint8_t guid_type, archive* archive, bool autonumber, data_file_format::types format, uint8_t log_level);

bool write_classification_archive(boost::shared_ptr<const lib3dv::classification> classificat, uint32_t guid, archive* archive, bool autonumber, data_file_format::types format, uint8_t log_level);
#endif

bool write_image_file(boost::shared_ptr<const lib3dv::image> image, uint32_t guid, uint8_t guid_type, const boost::filesystem::path& path, bool autonumber, uint8_t log_level);

bool write_terrain_file(boost::shared_ptr<const lib3dv::terrain> terrain, uint32_t guid, uint8_t guid_type, const boost::filesystem::path& path, bool autonumber, data_file_format::types format, uint8_t log_level);

bool write_obstacles_file(boost::shared_ptr<const std::vector<lib3dv::obstacle> > obstacles, uint32_t guid, uint8_t guid_type, const boost::filesystem::path& path, bool autonumber, data_file_format::types format, uint8_t log_level);

bool write_motion_file(boost::shared_ptr<const lib3dv::motion> motion, uint32_t guid, uint8_t guid_type, const boost::filesystem::path& path, bool autonumber, data_file_format::types format, uint8_t log_level);

bool write_classification_file(boost::shared_ptr<const lib3dv::classification> motion, uint32_t guid, uint8_t guid_type, const boost::filesystem::path& path, bool autonumber, data_file_format::types format, uint8_t log_level);

#endif
