/* 3dv-client/utils.cc
 *
 * Copyright (C) 2013 VisLab
 *
 * This file is part of lib3dv; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include "utils.h"

#ifdef __GNUC__

#include <errno.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netdb.h>
#include <ifaddrs.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include <iostream>

std::vector<boost::asio::ip::address> list_interface_addresses()
{
    std::vector<boost::asio::ip::address> result;
    
    ifaddrs* ifaddr;

    if(getifaddrs(&ifaddr) == -1)
    {
        std::cerr << "[EE] 3dv-client: " << strerror(errno) << std::endl;
        return result;
    }

    for(ifaddrs* ifa = ifaddr; ifa != NULL; ifa = ifa->ifa_next)
    {
        if(ifa->ifa_addr == NULL)
            continue;

        int family = ifa->ifa_addr->sa_family;

        if(family == AF_INET /*|| family == AF_INET6*/)
        {
            char host[NI_MAXHOST];

            int s = getnameinfo(ifa->ifa_addr, (family == AF_INET) ? sizeof(struct sockaddr_in) : sizeof(struct sockaddr_in6), host, NI_MAXHOST, NULL, 0, NI_NUMERICHOST);

            if(s)
            {
                std::cerr << "[EE] 3dv-client: " << gai_strerror(s) << std::endl;                
                continue;
            }
            
            boost::asio::ip::address address = boost::asio::ip::address::from_string(host);
            
            if(!address.is_loopback())
                result.push_back(boost::asio::ip::address::from_string(host));
        }
    }

    freeifaddrs(ifaddr);
    
    return result;
}

#elif _WIN32

#include <winsock2.h>
#include <iphlpapi.h>

std::vector<boost::asio::ip::address> list_interface_addresses()
{
    std::vector<boost::asio::ip::address> result;
    IP_ADAPTER_ADDRESSES* adapter_addresses(NULL);
    IP_ADAPTER_ADDRESSES* adapter(NULL);

    // Start with a 16 KB buffer and resize if needed -
    // multiple attempts in case interfaces change while
    // we are in the middle of querying them.
    DWORD adapter_addresses_buffer_size = 16 * 1024;
    for(int attempts = 0; attempts != 3; ++attempts)
    {
        adapter_addresses = (IP_ADAPTER_ADDRESSES*)malloc(adapter_addresses_buffer_size);
        assert(adapter_addresses);

        DWORD error = ::GetAdaptersAddresses(AF_UNSPEC, GAA_FLAG_SKIP_ANYCAST | GAA_FLAG_SKIP_MULTICAST | GAA_FLAG_SKIP_DNS_SERVER | GAA_FLAG_SKIP_FRIENDLY_NAME, NULL,  adapter_addresses, &adapter_addresses_buffer_size);

        if(error == ERROR_SUCCESS)
            break;
        else if(error == ERROR_BUFFER_OVERFLOW)
        {
            // Try again with the new size
            free(adapter_addresses);
            adapter_addresses = NULL;
            continue;
        }
    }

    // Iterate through all of the adapters
    for(adapter = adapter_addresses; NULL != adapter; adapter = adapter->Next)
    {
        // Skip loopback adapters
        if (IF_TYPE_SOFTWARE_LOOPBACK == adapter->IfType)
            continue;

        // Parse all IPv4 addresses
        for(IP_ADAPTER_UNICAST_ADDRESS* address = adapter->FirstUnicastAddress; address != NULL; address = address->Next)
        {
            if(address->Address.lpSockaddr->sa_family == AF_INET)
            {
                SOCKADDR_IN* ipv4 = reinterpret_cast<SOCKADDR_IN*>(address->Address.lpSockaddr);
                result.push_back(boost::asio::ip::address_v4(ntohl(ipv4->sin_addr.s_addr)));
            }
        }
    }

    free(adapter_addresses);
    adapter_addresses = NULL;

    return result;
}

#endif

#ifdef PCL

#include "display.h"

void image_callback_display(boost::shared_ptr<const lib3dv::image> image, uint32_t guid, display& display, uint8_t log_level)
{
    display.update(image);
}

void terrain_callback_display(boost::shared_ptr<const lib3dv::terrain> terrain, uint32_t guid, display& display, uint8_t log_level)
{
    display.update(terrain);
}

void obstacles_callback_display(boost::shared_ptr<const std::vector<lib3dv::obstacle> > obstacles, uint32_t guid, display& display, uint8_t log_level)
{
    display.update(obstacles);
}

void motion_callback_display(boost::shared_ptr<const lib3dv::motion> motion, uint32_t guid, display& display, uint8_t log_level)
{
    display.update(motion);
}

void classification_callback_display(boost::shared_ptr<const lib3dv::classification> classificat, uint32_t guid, display& display, uint8_t log_level)
{
    display.update(classificat);
}

#endif

#include "disk_writer.h"
#include "display.h"

#ifdef PCL
void signal_handler(const boost::system::error_code& error, int signal_number, boost::shared_ptr<disk_writer>& disk_writer, boost::shared_ptr<display>& display, boost::asio::io_service& sighandler)
{
    std::cout << std::endl;
    
    sighandler.stop();
}

void timeout_callback(boost::shared_ptr<disk_writer>& disk_writer, boost::shared_ptr<display>& display, boost::asio::io_service& sighandler)
{
    std::cout << std::endl << "[EE] 3dv-client: device error NETWORK_TIMEOUT" << std::endl;
    
    sighandler.stop();
}
#else
void signal_handler(const boost::system::error_code& error, int signal_number, boost::shared_ptr<disk_writer>& disk_writer, boost::asio::io_service& sighandler)
{
    std::cout << std::endl;

    sighandler.stop();
}

void timeout_callback(boost::shared_ptr<disk_writer>& disk_writer, boost::asio::io_service& sighandler)
{
    std::cout << std::endl << "[EE] 3dv-client: device error NETWORK_TIMEOUT" << std::endl;
    
    sighandler.stop();
}
#endif
