/* 3dv-client/device.cc
 *
 * Copyright (C) 2013 VisLab
 *
 * This file is part of 3dv-client; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <lib3dv/device.h>

#include <lib3dv/detail/device_impl.h>

#include <iomanip>
#include <typeinfo>


namespace lib3dv
{
    const unsigned short device::DEFAULT_PORT = 2048;
    const boost::posix_time::time_duration device::DEFAULT_TIMEOUT = boost::posix_time::milliseconds(500);
    
    template<typename T> inline void print(std::ostream& output, const T& value) { output << value; }
    template<> inline void print<uint8_t>(std::ostream& output, const uint8_t& value) { output << static_cast<uint32_t>(value); }
    template<> inline void print<int8_t>(std::ostream& output, const int8_t& value) { output << static_cast<int32_t>(value); }
    
    template<typename T>
    void print(std::ostream& output, const std::string& type, const device::property& property)
    {
        output << "0x" << std::hex << std::setfill('0') << std::setw(3) << property.m_address
                << std::dec << "    " << type << (property.m_readonly ? "    r     " : "    rw    ")
                << property.m_name << " = ";

        print(output, boost::any_cast<T>(property.m_value));
                
        output << " ";

        for(unsigned int a = 0; a < property.m_attributes.size(); ++a)
        {
            output << " [";
            print(output, boost::any_cast<T>(property.m_attributes[a].first)); 
            
            if(property.m_attributes[a].second.empty())
                output << "]";
            else
                output << " -> " << property.m_attributes[a].second << "]";
        }
    }
    
    std::ostream& operator<< (std::ostream& output, const device::property& property)
    {
        const std::type_info& type = property.m_value.type();
        
        if(type == typeid(bool)) print<bool>(output, "bool   ", property);
        else if(type == typeid(int64_t)) print<int64_t>(output, "int64  ", property);
        else if(type == typeid(int32_t)) print<int32_t>(output, "int32  ", property);
        else if(type == typeid(int16_t)) print<int16_t>(output, "int16  ", property);
        else if(type == typeid(int8_t)) print<int8_t>(output, "int8   ", property);
        else if(type == typeid(uint64_t)) print<uint64_t>(output, "uint64 ", property);
        else if(type == typeid(uint32_t)) print<uint32_t>(output, "uint32 ", property);
        else if(type == typeid(uint16_t)) print<uint16_t>(output, "uint16 ", property);
        else if(type == typeid(uint8_t)) print<uint8_t>(output, "uint8  ", property);
        else if(type == typeid(float)) print<float>(output, "float32", property);
        else if(type == typeid(double)) print<double>(output, "float64", property);
        else if(type == typeid(std::string)) print<std::string>(output, "string ", property);
        
        return output;   
    }
    
    std::vector<device> device::enumerate(const boost::asio::ip::address_v4& local_address, uint8_t log_level, error& error, const boost::posix_time::time_duration& timeout, unsigned short remote_port) { return detail::device_impl::enumerate(local_address, remote_port, log_level, error, timeout); }

    bool device::valid() const { return m_impl->valid(); }
    
    device::status::types device::status() const { return m_impl->status(); }
    
    void device::log_level(uint8_t level) { m_impl->m_log_level = level; }
    
    uint8_t device::log_level() const { return m_impl->m_log_level; }
    
    void device::timeout(const boost::posix_time::time_duration& timeout) { m_impl->m_timeout = timeout; }

    boost::posix_time::time_duration device::timeout() const { return m_impl->m_timeout; }
    
    const boost::uuids::uuid& device::guid() const { return m_impl->guid(); }
    
    const std::bitset<device::capability::NUM>& device::capabilities() const { return m_impl->capabilities(); }
    
    const device::version_info& device::version() const { return m_impl->version(); }
    
    void device::start_transmission(error& error) { if(m_impl->valid()) m_impl->start_transmission(error); else error = lib3dv::error::NETWORK_FAILURE; }
            
    void device::stop_transmission(error& error) { if(m_impl->valid()) m_impl->stop_transmission(error); else error = lib3dv::error::NETWORK_FAILURE; }
    
    void device::save_properties(error& error) { if(m_impl->valid()) m_impl->save_properties(error); else error = lib3dv::error::NETWORK_FAILURE; }

    void device::reset_properties(error& error) { if(m_impl->valid()) m_impl->reset_properties(error); else error = lib3dv::error::NETWORK_FAILURE; }
    
    void device::poweroff(error& error) { if(m_impl->valid()) m_impl->poweroff(error); else error = lib3dv::error::NETWORK_FAILURE; }
    
    void device::reboot(error& error) { if(m_impl->valid()) m_impl->reboot(error); else error = lib3dv::error::NETWORK_FAILURE; }
    
    std::vector<device::property> device::enumerate_properties(error& error) { if(m_impl->valid()) return m_impl->enumerate_properties(error); else error = lib3dv::error::NETWORK_FAILURE; }
    
    boost::any device::get_property_value(uint16_t address, error& error) { if(m_impl->valid()) return m_impl->get_property_value(address, error); else error = lib3dv::error::NETWORK_FAILURE; }
            
    void device::set_property_value(uint16_t address, const boost::any& value, error& error) { if(m_impl->valid()) m_impl->set_property_value(address, value, error); else error = lib3dv::error::NETWORK_FAILURE; }
    
    uint64_t device::connect_image_callback(const boost::function<void (boost::shared_ptr<const image>, uint32_t)>& function) { return m_impl->connect_image_callback(function); }
    
    uint64_t device::connect_terrain_callback(const boost::function<void (boost::shared_ptr<const terrain>, uint32_t)>& function) { return m_impl->connect_terrain_callback(function); }
    
    uint64_t device::connect_obstacles_callback(const boost::function<void (boost::shared_ptr<const std::vector<obstacle> >, uint32_t)>& function) { return m_impl->connect_obstacles_callback(function); }
    
    uint64_t device::connect_motion_callback(const boost::function<void (boost::shared_ptr<const motion>, uint32_t)>& function) { return m_impl->connect_motion_callback(function); }

    uint64_t device::connect_classification_callback(const boost::function<void (boost::shared_ptr<const lib3dv::classification>, uint32_t)>& function) { return m_impl->connect_classification_callback(function); }

    LIB3DV_DEPRECATED uint64_t device::connect_image_callback(const boost::function<void (boost::shared_ptr<image>, uint32_t)>& function) { return m_impl->connect_image_callback(function); }

    LIB3DV_DEPRECATED uint64_t device::connect_terrain_callback(const boost::function<void (boost::shared_ptr<terrain>, uint32_t)>& function) { return m_impl->connect_terrain_callback(function); }

    LIB3DV_DEPRECATED uint64_t device::connect_obstacles_callback(const boost::function<void (boost::shared_ptr<std::vector<obstacle> >, uint32_t)>& function) { return m_impl->connect_obstacles_callback(function); }

    LIB3DV_DEPRECATED uint64_t device::connect_motion_callback(const boost::function<void (boost::shared_ptr<motion>, uint32_t)>& function) { return m_impl->connect_motion_callback(function); }

    LIB3DV_DEPRECATED uint64_t device::connect_classification_callback(const boost::function<void (boost::shared_ptr<lib3dv::classification>, uint32_t)>& function) { return m_impl->connect_classification_callback(function); }

    uint64_t device::connect_timeout_callback(const boost::function<void (void)>& function) { return m_impl->connect_timeout_callback(function); }

    void device::disconnect_callback(uint64_t id) { m_impl->disconnect_callback(id); }
    
    std::ostream& operator<< (std::ostream& output, const device& device)
    {
        return output << *(device.m_impl);
    }
}
