#ifndef LIB3DV_DEVICE_H
#define LIB3DV_DEVICE_H

/* lib3dv/device.h
 *
 * Copyright (C) 2013 VisLab
 *
 * This file is part of lib3dv; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <lib3dv/error.h>
#include <lib3dv/image.h>
#include <lib3dv/obstacle.h>
#include <lib3dv/motion.h>
#include <lib3dv/classification.h>
#include <lib3dv/terrain.h>

#include <lib3dv/3dv_export.h>

#include <boost/any.hpp>
#include <boost/asio/ip/address_v4.hpp>
#include <boost/function.hpp>
#include <boost/date_time/time_duration.hpp>
#include <boost/uuid/uuid.hpp>
#include <boost/shared_ptr.hpp>

#include <bitset>
#include <iostream>
#include <string>
#include <utility>
#include <vector>

namespace lib3dv
{
    namespace detail { class device_impl; }
    
    /**
     * @brief This class represents a 3DV device connected to the local machine.
     * 
     */
    class LIB3DV_EXPORT device
    {
        public:
        
            /**
             * @brief Supported mapping and detection capabilities.
             * 
             */
            struct capability
            {
                enum types
                {
                    DEPTH_MAPPING = 0,          ///< The device provides depth data.
                    TERRAIN_MAPPING,            ///< The device provides the terrain reconstruction.
                    OBSTACLE_DETECTION,         ///< The device provides a list of detected obstacles.
                    BASIC_OBSTACLE_DETECTION,   ///< The device provides a list of detected obstacles, under flat ground assumption.
                    MOTION_ESTIMATION,          ///< The device provides the motion estimation matrix.
                    CLASSIFICATION,             ///< The device provides the classification candidate.
                    NUM
                };
            };
            
            /**
             * @brief Device version information.
             * 
             */
            struct version_info
            {
                enum types
                {
                    MAJOR = 0,      ///< Major version.
                    MINOR,          ///< Minor version.
                    STEP            ///< Patch level.
                };
                
                uint8_t m_protocol;         ///< Network protocol version.
                uint8_t m_framework[3];     ///< Framework version.
                uint8_t m_application[3];   ///< 3DV application version.
            };
            
            /**
             * @brief Device status information.
             * 
             */
            struct status
            {
                enum types
                {
                    ONLINE = 0,         ///< The device is available.
                    TRANSMITTING,       ///< The device is streaming data to this lib3dv::device instance.
                    OFFLINE,            ///< The device is no longer connected.
                    NUM,
                };
            };
            
            /**
             * @brief A configuration parameter.
             * 
             * A property is a generic container hiding the actual data type within a boost::any. In order to recover the type information typeid() must be used.
             * 
             * @note Currently supported data types are bool, [u]int64_t, [u]int32_t, [u]int16_t, [u]int8_t, float, double, std::string.
             * 
             * The following example shows how to determine the property value type:
             * 
               @code
               lib3dv::error error;
               lib3dv::device::property property = device.enumerate_properties(error).front();
               const boost::any& value = property.m_value;
               const std::type_info& type = value.type();
               
               if(type == typeid(bool)) std::cout << boost::any_cast<bool>(value) << std::endl;
               else if(type == typeid(int64_t)) std::cout << boost::any_cast<int64_t>(value) << std::endl;
               else if(type == typeid(int32_t)) std::cout << boost::any_cast<int32_t>(value) << std::endl;
               else if(type == typeid(int32_t)) std::cout << boost::any_cast<int32_t>(value) << std::endl;
               else if(type == typeid(int16_t)) std::cout << boost::any_cast<int16_t>(value) << std::endl;
               else if(type == typeid(int8_t)) std::cout << static_cast<int32_t>(boost::any_cast<int8_t>(value)) << std::endl;
               else if(type == typeid(uint64_t)) std::cout << boost::any_cast<uint64_t>(value) << std::endl;
               else if(type == typeid(uint32_t)) std::cout << boost::any_cast<uint32_t>(value) << std::endl;
               else if(type == typeid(uint32_t)) std::cout << boost::any_cast<uint32_t>(value) << std::endl;
               else if(type == typeid(uint16_t)) std::cout << boost::any_cast<uint16_t>(value) << std::endl;
               else if(type == typeid(uint8_t)) std::cout << static_cast<uint32_t>(boost::any_cast<uint8_t>(value)) << std::endl;
               else if(type == typeid(float)) std::cout << boost::any_cast<float>(value) << std::endl;
               else if(type == typeid(double)) std::cout << boost::any_cast<double>(value) << std::endl;
               else if(type == typeid(std::string)) std::cout << boost::any_cast<std::string>(value) << std::endl;
               @endcode
             */
            struct property
            {
                /**
                 * @brief The parameter category.
                 * 
                 * The category determines the property attributes, and is useful to generate proper UI controls.
                 */
                enum category
                {
                    VALUE = 0,  ///< Basic property with no constraints.
                    RANGE,      ///< Property values must fall in a determined interval. lib3dv::device::property::m_attributes will contain the "min", "max" and "step" entries.
                    SELECTION   ///< Property value must be chosen among those listed in lib3dv::device::property::m_attributes. Each entry can optionally provide a description.
                };
                
                uint16_t m_address;     ///< Property unique address.
                std::string m_name;     ///< Property name.
                bool m_readonly;        ///< Whether the property can be written to. @see lib3dv::error::PROPERTY_READONLY
                category m_category;    ///< Property category.
                boost::any m_value;     ///< Property value.
                std::vector<std::pair<boost::any, std::string> > m_attributes; ///< List of attributes.
                
                /**
                 * @brief Convenience display function.
                 * 
                 * @param output The std::ostream to write to
                 * @param property The property to display
                 * @return std::ostream&
                 */                
                friend LIB3DV_EXPORT std::ostream& operator<< (std::ostream& output, const device::property& property);
            };

            static const unsigned short DEFAULT_PORT; ///< Default UDP port to connect to.
            
            static const boost::posix_time::time_duration DEFAULT_TIMEOUT; ///< Default connection timeout.
            
            /**
             * @brief List available devices
             * 
             * A broadcast UDP discovery message is sent through the specified local interface, and a lib3dv::device object is associated to every 3DV unit responding before the timeout expires.
             * @note The call is blocking,
             * 
             * @param [in] local_address The local IP address to bing to. Must belong to the same subnet of the 3DV devices to connect to.
             * @param [in] log_level The log messages verbosity level.
             * @param [out] error Result of operation.
             * @param [in] timeout How long the function should wait for 3DV devices to answer to the discovery message. Defaults to lib3dv::device::DEFAULT_TIMEOUT.
             * @param [in] remote_port The remote port to connect to. Defaults to lib3dv::device::DEFAULT_PORT.
             * @return std::vector< lib3dv::device > The detected devices.
             * 
             * Example usage:
               @code
               std::string address = "192.168.0.1";
               unsigned int log_level = 1;
               lib3dv::error error;
             
               std::vector<lib3dv::device> devices = lib3dv::device::enumerate(boost::asio::ip::address_v4::from_string(address), log_level, error);
               @endcode
             */
            static std::vector<device> enumerate(const boost::asio::ip::address_v4& local_address, uint8_t log_level, error& error, const boost::posix_time::time_duration& timeout = DEFAULT_TIMEOUT, unsigned short remote_port = DEFAULT_PORT);

            /**
             * @brief Whether the device is still in a valid state.
             * 
             * @return bool (lib3dv::device::status() == lib3dv::device::status::ONLINE) || (lib3dv::device::status() == lib3dv::device::status::TRANSMITTING).
             */
            bool valid() const;
            
            /**
             * @brief Whether the device is still in a valid state.
             * 
             * @return bool false if the physical device is no longer reachable, true otherwise.
             */
            status::types status() const;
            
            /**
             * @brief Set the log messages verbosity level.
             * 
             * @param [in] level The verbosity level.
             * @return void
             */
            void log_level(uint8_t level);

            /**
             * @brief Get the current log messages verbosity level.
             * 
             * @return uint8_t The verbosity level.
             */
            uint8_t log_level() const;

            /**
             * @brief Set the network connection timeout threshold.
             * 
             * After a timeout, any method invocation requiring to communicate with the device will fail with a lib3dv::error::NETWORK_FAILURE error, and no data handlers will be triggered. @see lib3dv::device::valid()
             * 
             * @param [in] timeout The network connection timeout threshold.
             * @return void
             */
            void timeout(const boost::posix_time::time_duration& timeout);
            
            /**
             * @brief Get the current network connection timeout threshold.
             * 
             * @return boost::posix_time::time_duration The network connection timeout threshold.
             */
            boost::posix_time::time_duration timeout() const;
            
            /**
             * @brief Get the device unique ID.
             * 
             * @return const boost::uuids::uuid& The device unique ID.
             */
            const boost::uuids::uuid& guid() const;
            
            /**
             * @brief Enumerate the device capabilities. @see lib3dv::device::capability
             * 
             * @return const std::bitset< lib3dv::device::capability::NUM >&
             */
            const std::bitset<device::capability::NUM>& capabilities() const;
            
            /**
             * @brief Get version information
             * 
             * @return const lib3dv::device::version_info&
             */
            const version_info& version() const;
            
            /**
             * @brief Enable data transmission on the device.
             * 
             * Notify the device that it should start transmitting data. If transmission is already on it should first be stopped by calling lib3dv::device::stop_transmission, otherwise the call is ignored.
             * @note The call is blocking,             
             * @see lib3dv::device::connect_image_callback
             * @see lib3dv::device::connect_terrain_callback
             * @see lib3dv::device::connect_obstacles_callback
             * 
             * @param [out] error Result of operation.
             * @return void
             */
            void start_transmission(error& error);
            
            /**
             * @brief Disable data transmission on the device.
             * 
             * Notify the device that it should stop transmitting data. This method has effect even if the data transmission has been initiated by a different client.
             * @note The call is blocking.
             * @see lib3dv::device::disconnect_callback
             * 
             * @param [out] error Result of operation.
             * @return void
             */
            void stop_transmission(error& error);            
            
            /**
             * @brief Enumerate the device properties.
             * @note The call is blocking.
             * @see lib3dv::device::property
             * 
             * @param [out] error Result of operation.
             * @return std::vector< lib3dv::device::property > The device properties.
             */
            std::vector<device::property> enumerate_properties(error& error);
            
            /**
             * @brief Read a property value from the device.
             * @note The call is blocking.
             * @see lib3dv::device::property
             * 
             * @param [in] address The property address.
             * @param [out] error Result of operation.
             * @return boost::any The property value.
             */
            boost::any get_property_value(uint16_t address, error& error);
            
            /**
             * @brief Store a property value on the device.
             * 
             * The given value is set on the device; however, changes are not permanently stored until lib3dv::device::save_properties is called.
             * @note The call is blocking.
             * @see lib3dv::device::property
             * @see lib3dv::device::save_properties
             * 
             * @param [in] address The property address.
             * @param [in] value The value to store. If the correct type is not supplied an error will be generated.
             * @param [out] error Result of operation.
             * @return void
             */            
            void set_property_value(uint16_t address, const boost::any& value, error& error);
            
            /**
             * @brief Permanently store the current configuration on the device.
             * @note The call is blocking.
             * 
             * @param [out] error Result of operation.
             * @return void
             */
            void save_properties(error& error);

            /**
             * @brief Resets the device configuration to the default values.
             * @note The call is blocking.
             * @note A reboot is required to make the changes effective.
             * 
             * @param [out] error Result of operation.
             * @return void
             */            
            void reset_properties(error& error);
            
            /**
             * @brief Shuts down the device.
             * @note The call is blocking.
             * 
             * @param [out] error Result of operation.
             * @return void
             */
            void poweroff(error& error);
            
            /**
             * @brief Reboots the device.
             * @note The call is blocking.
             * 
             * @param [out] error Result of operation.
             * @return void
             */            
            void reboot(error& error);
            
            /**
             * @brief Register an image completion handler.
             * 
             * The supplied user-defined function will be called each time a full image is received.
             * @note The completion handler is executed in the network handling thread, so it should not perform any actual processing.
             * Instead, it should just copy over the provided shared pointer and notify a worker thread.
             * @note No guarantee is made on the images ordering, or that any given image will be completed at all.
             * 
             * @param [in] function The user-defined callback function. Its signature must be  <tt> void (boost::shared_ptr<cosnt lib3dv::image> image, uint32_t guid)</tt>
             * @return uint64_t An opaque handle that can be passed to lib3dv::device::disconnect_callback to unregister the handler.
             *          
             * Example usage, exploiting boost::asio to achieve deferred processing on a separate thread.
               @code
               #include <boost/asio/io_service.hpp>
               #include <boost/bind.hpp>
               #include <boost/ref.hpp>
               #include <boost/thread.hpp>
               
               // actual processing happens here 
               void process_image(boost::shared_ptr<cosnt lib3dv::image> image, uint32_t guid)
               {
                   // do stuff
               }
               
               // lightweight completion handler
               void image_completion_handler(boost::asio::io_service& io_service, boost::shared_ptr<const lib3dv::image> image, uint32_t guid)
               {
                   io_service.post(boost::bind(process_image, image, guid)); // process_image() is scheduled for execution on the io_service::run() thread
               }
               
               ...
               
               boost::asio::io_service processing_io_service;
               boost::asio::io_service::work processing_io_service_work(processing_io_service); // this keeps the io_service from exiting when there are no handlers to execute
               boost::thread processing_io_service_thread = boost::thread(boost::bind(&boost::asio::io_service::run, &processing_io_service)); // start the processing thread
               
               device.connect_image_callback(boost::bind(image_completion_handler, boost::ref(processing_io_service), _1, _2)); // subscribe for images notifications
               
               lib3dv::device::error error;
               
               device.start_transmission(error); // notify the device it can start sending data
               @endcode
             */
            uint64_t connect_image_callback(const boost::function<void (boost::shared_ptr<const image>, uint32_t)>& function);

            LIB3DV_DEPRECATED uint64_t connect_image_callback(const boost::function<void (boost::shared_ptr<image>, uint32_t)>& function);

            /**
             * @brief Register a terrain completion handler.
             * 
             * The supplied user-defined function will be called each time a terrain reconstruction is received.
             * @note The completion handler is executed in the network handling thread, so it should not perform any actual processing.
             * Instead, it should just copy over the provided shared pointer and notify a worker thread.
             * @note No guarantee is made on the images ordering, or that any given image will be completed at all.
             * @see lib3dv::device::connect_image_callback
             * 
             * @param [in] function The user-defined callback function. Its signature must be  <tt> void (boost::shared_ptr<const lib3dv::terrain> terrain, uint32_t guid)</tt>.
             * The @c guid parameter is the same as that of the depth map that has been processed to generate the current terrain estimation.
             * @return uint64_t An opaque handle that can be passed to lib3dv::device::disconnect_callback to unregister the handler.
             */
            uint64_t connect_terrain_callback(const boost::function<void (boost::shared_ptr<const terrain>, uint32_t)>& function);

            LIB3DV_DEPRECATED uint64_t connect_terrain_callback(const boost::function<void (boost::shared_ptr<terrain>, uint32_t)>& function);

            /**
             * @brief Register an obstacles set completion handler.
             * 
             * The supplied user-defined function will be called each time a set of obstacles is received.
             * @note The completion handler is executed in the network handling thread, so it should not perform any actual processing.
             * Instead, it should just copy over the provided shared pointer and notify a worker thread.
             * @note No guarantee is made on the images ordering, or that any given image will be completed at all.
             * @see lib3dv::device::connect_image_callback
             * 
             * @param [in] function The user-defined callback function. Its signature must be <tt> void (boost::shared_ptr<const std::vector<lib3dv::obstacle> > obstacles, uint32_t guid)</tt>.
             * The @c guid parameter is the same as that of the depth map that has been processed to generate the current set of obstacles.
             * @return uint64_t An opaque handle that can be passed to lib3dv::device::disconnect_callback to unregister the handler.
             */            
            uint64_t connect_obstacles_callback(const boost::function<void (boost::shared_ptr<const std::vector<obstacle> >, uint32_t)>& function);

            LIB3DV_DEPRECATED uint64_t connect_obstacles_callback(const boost::function<void (boost::shared_ptr<std::vector<obstacle> >, uint32_t)>& function);

            /**
             * @brief Register a motion completion handler.
             *
             * The supplied user-defined function will be called each time a motion estimation is received.
             * @note The completion handler is executed in the network handling thread, so it should not perform any actual processing.
             * Instead, it should just copy over the provided shared pointer and notify a worker thread.
             * @note No guarantee is made on the images ordering, or that any given image will be completed at all.
             * @see lib3dv::device::connect_image_callback
             *
             * @param [in] function The user-defined callback function. Its signature must be  <tt> void (boost::shared_ptr<const lib3dv::motion> motion, uint32_t guid)</tt>.
             * The @c guid parameter is the same as that of the depth map that has been processed to generate the current motion estimation.
             * @return uint64_t An opaque handle that can be passed to lib3dv::device::disconnect_callback to unregister the handler.
             */
            uint64_t connect_motion_callback(const boost::function<void (boost::shared_ptr<const motion>, uint32_t)>& function);

            LIB3DV_DEPRECATED uint64_t connect_motion_callback(const boost::function<void (boost::shared_ptr<motion>, uint32_t)>& function);

            /**
             * @brief Register a classification completion handler.
             *
             * The supplied user-defined function will be called each time a classification is received.
             * @note The completion handler is executed in the network handling thread, so it should not perform any actual processing.
             * Instead, it should just copy over the provided shared pointer and notify a worker thread.
             * @note No guarantee is made on the images ordering, or that any given image will be completed at all.
             * @see lib3dv::device::connect_image_callback
             *
             * @param [in] function The user-defined callback function. Its signature must be  <tt> void (boost::shared_ptr<const lib3dv::classification> classificat, uint32_t guid)</tt>.
             * The @c guid parameter is the same as that of the depth map that has been processed to generate the current classification.
             * @return uint64_t An opaque handle that can be passed to lib3dv::device::disconnect_callback to unregister the handler.
             */
            uint64_t connect_classification_callback(const boost::function<void (boost::shared_ptr<const classification>, uint32_t)>& function);

            LIB3DV_DEPRECATED uint64_t connect_classification_callback(const boost::function<void (boost::shared_ptr<classification>, uint32_t)>& function);

            /**
             * @brief Register a timeout handler.
             *
             * The supplied user-defined function will be called each time a timeout condition is reached.
             * @see lib3dv::device::timeout
             * @see lib3dv::device::valid
             * 
             * @param [in] function The user-defined callback function. Its signature must be <tt>void (void)</tt>.
             * @return uint64_t An opaque handle that can be passed to lib3dv::device::disconnect_callback to unregister the handler.
             */
            uint64_t connect_timeout_callback(const boost::function<void (void)>& function);
            
            /**
             * @brief Unregister a given handler.
             * 
             * @param [in] id The handler ID. No action is performed if it is invalid.
             * @return void
             */
            void disconnect_callback(uint64_t id);
                        
            /**
             * @brief Generate a textual representation of a given device and insert it into an std::ostream
             * 
             * @param [out] output The std::ostream to use.
             * @param [in] device The device to describe.
             * @return std::ostream& The modified std::ostream
             */
            friend LIB3DV_EXPORT std::ostream& operator<< (std::ostream& output, const device& device);
            
        private:
            
            device(){}
            
            friend device create(boost::shared_ptr<detail::device_impl>);

            boost::shared_ptr<detail::device_impl> m_impl;
    };
}

#endif
