#ifndef LIB3DV_IMAGE_H
#define LIB3DV_IMAGE_H

/* lib3dv/image.h
 *
 * Copyright (C) 2013 VisLab
 *
 * This file is part of lib3dv; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <boost/date_time/posix_time/posix_time.hpp>

#include <stdint.h>
#include <vector>

namespace lib3dv
{
    /**
     * @brief An image received from the device.
     * 
     */
    struct image
    {
        /**
        * @brief Supported image formats.
        **/
        struct format
        {
            enum types
            {
                NONE = 0,
                MONO8,              ///< Monochrome 8 bits.
                MONO16,             ///< Monochrome 16 bits.
                RGB24,              ///< RGB, 8 bits per channel.
                RGB48,              ///< RGB, 16 bits per channel.
                BAYER8_BGGR,        ///< BGGR Bayer pattern, 8 bits per channel.
                BAYER8_GRBG,        ///< GRBG Bayer pattern, 8 bits per channel.
                BAYER8_RGGB,        ///< RGGB Bayer pattern, 8 bits per channel.
                BAYER8_GBRG,        ///< GBRG Bayer pattern, 8 bits per channel.
                BAYER16_BGGR,       ///< BGGR Bayer pattern, 16 bits per channel.
                BAYER16_GRBG,       ///< GRBG Bayer pattern, 16 bits per channel.
                BAYER16_RGGB,       ///< RGGB Bayer pattern, 16 bits per channel.
                BAYER16_GBRG,       ///< GBRG Bayer pattern, 16 bits per channel.
                FLOAT32,            ///< Single channel 32 bits floating point.
                FLOAT64,            ///< Single channel 64 bits floating point.
                NUM
            };
        };
        
        /**
        * @brief Supported image types.
        * @note DSI image values are encoded as 16-bit little-endian fixed-point integers, so that the actual disparity value can be computed as
        * 
          @code
          std::vector<float> disparities(dsi->m_width * dsi->m_height);
         
          for(unsigned int d = 0; d < dsi->m_width * dsi->m_height; ++d)
            disparities[d] = static_cast<const uint16_t*>(dsi->m_buffer.data())[d] / 255.0f
          @endcode
        **/
        struct type
        {
            enum types
            {
                NONE = 0,
                LEFT_RECTIFIED,     ///< Left rectified image.
                RIGHT_RECTIFIED,    ///< Right rectified image.
                LEFT_RAW,           ///< Left raw image.
                RIGHT_RAW,          ///< Right raw image.
                DSI,                ///< Depth map
                NUM
            };
        };
        
        uint16_t m_width;                       ///< Image width [px].
        uint16_t m_height;                      ///< Image height [px].
        uint8_t m_channels;                     ///< Image channels.
        uint8_t m_bpp;                          ///< Image bits per pixel [bpp].
        type::types m_type;                     ///< Image type. @see lib3dv::image::type
        format::types m_format;                 ///< Image format. @see lib3dv::image::format
        boost::posix_time::ptime m_timestamp;   ///< Image timestamp.
        std::vector<uint8_t> m_buffer;          ///< Image data buffer.
    };
}

#endif
