/* lib3dv/detail/device_impl.cc
 *
 * Copyright (C) 2013 VisLab
 *
 * This file is part of lib3dv; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */


#include <lib3dv/detail/device_impl.h>

#include <boost/asio/deadline_timer.hpp>
#include <boost/bind/protect.hpp>
#include <boost/ref.hpp>
#include <boost/system/error_code.hpp>
#include <boost/uuid/uuid_io.hpp>

#include <cstring>
#include <iostream>
#include <vector>

namespace lib3dv
{
    void on_deadline(boost::asio::deadline_timer& deadline,
                     boost::asio::ip::udp::socket& socket,
                     uint8_t log_level,
                     bool wait,
                     boost::asio::io_service& io_service,
                     const boost::function<void (void)>& cleanup,
                     const boost::system::error_code& error)
    {
        if(!error) // can be a direct invocation, or triggered by an expired timer
        {
            if(deadline.expires_at() < boost::asio::deadline_timer::traits_type::now()) // expired timer
            {
                if(log_level > 0) std::cout << "[II] lib3dv: deadline expired, cleaning up..." << std::endl;
            
                cleanup();
            }
            else // direct (first) invocation
            {
                deadline.async_wait(boost::bind(on_deadline, boost::ref(deadline), boost::ref(socket), log_level, wait, boost::ref(io_service), cleanup, _1));
            }
        }
        else if(error == boost::asio::error::operation_aborted) // can be triggered by cancel() or expires_from_now()
        {
            if(log_level > 1) std::cout << "[II] lib3dv: deadline canceled" << std::endl; // cancel(), nothing to do
            
            if(wait) // expires_from_now() we start to wait again
            {
                if(log_level > 1) std::cout << "[II] lib3dv: data deadline extended" << std::endl;
                
                deadline.async_wait(boost::bind(on_deadline, boost::ref(deadline), boost::ref(socket), log_level, wait, boost::ref(io_service), cleanup, _1));
            }
        }
        else // another error occourred
            std::cerr << "[EE] lib3dv: " << error.message() << std::endl;
    }
    
    lib3dv::device create(boost::shared_ptr<lib3dv::detail::device_impl> impl)
    {
        lib3dv::device device;
        device.m_impl = impl;
        return device;
    }

    namespace detail
    {        
        device_impl::device_impl(const boost::asio::ip::address_v4& local_address,
                                 const boost::asio::ip::address_v4& remote_address, unsigned short remote_commands_port, unsigned short remote_data_port,
                                 const boost::uuids::uuid& guid,
                                 const std::bitset<device::capability::NUM>& capabilities,
                                 const device::version_info& version,
                                 uint8_t log_level,
                                 lib3dv::error& device_error) :
            m_log_level(log_level),
            m_timeout(device::DEFAULT_TIMEOUT),
            m_local_address(local_address),
            m_remote_address(remote_address),
            m_remote_commands_port(remote_commands_port),
            m_remote_data_port(remote_data_port),
            m_data_socket(m_data_socket_io_service),
            m_commands_socket(m_commands_socket_io_service),
            m_commands_deadline(m_commands_socket_io_service),
            m_data_deadline(m_data_socket_io_service),
            m_guid(guid),
            m_capabilities(capabilities),
            m_version(version),
            m_max_connection_id(0),
            m_status(device::status::ONLINE)
        {
            boost::system::error_code error;
            
            m_data_socket.open(boost::asio::ip::udp::v4(), error);
            
            if(error)
            {
                std::cerr << "[EE] lib3dv: " << error.message() << std::endl;                
                device_error = lib3dv::error::NETWORK_FAILURE;
            }
            
            m_commands_socket.open(boost::asio::ip::udp::v4(), error);

            if(error)
            {
                std::cerr << "[EE] lib3dv: " << error.message() << std::endl;                
                device_error = lib3dv::error::NETWORK_FAILURE;
            }
            
#ifdef _MSC_VER
            boost::asio::socket_base::receive_buffer_size option(8192 * 1024);
            m_data_socket.set_option(option);
#endif

            m_data_socket.set_option(boost::asio::socket_base::broadcast(true));
            m_data_socket.bind(boost::asio::ip::udp::endpoint(local_address, 0), error);
            
            if(error)
            {
                std::cerr << "[EE] lib3dv: " << error.message() << std::endl;                
                device_error = lib3dv::error::NETWORK_FAILURE;
            }

            m_commands_socket.set_option(boost::asio::socket_base::broadcast(true));
            m_commands_socket.bind(boost::asio::ip::udp::endpoint(local_address, 0), error);

            if(error)
            {
                std::cerr << "[EE] lib3dv: " << error.message() << std::endl;                
                device_error = lib3dv::error::NETWORK_FAILURE;
            }
            
            m_data_socket_io_service.post(boost::bind(&device_impl::on_data_received, this, boost::system::error_code(), 0));            
            m_data_socket_io_service_thread = boost::thread(boost::bind(&boost::asio::io_service::run, &m_data_socket_io_service));

            if(log_level > 0) std::cout << "[II] lib3dv: device " << m_guid << std::endl;
            if(log_level > 0) std::cout << "\tcontrol channel: " << m_commands_socket.local_endpoint() << " <-> " << m_remote_address << ':' << m_remote_commands_port << std::endl;
            if(log_level > 0) std::cout << "\tdata channel " << m_data_socket.local_endpoint() << " <-> " << m_remote_address << ':' << m_remote_data_port << std::endl;
            
            if(device_error != lib3dv::error::NONE)
                m_status = device::status::OFFLINE;
        }
        
        device_impl::~device_impl()
        {
            if(m_log_level > 0) std::cout << "[II] lib3dv: destructor called" << std::endl;            
            
            boost::system::error_code error;
            
            if(m_data_socket.is_open())
                m_data_socket.close(error);
            
            m_data_socket_io_service.stop();                        
            
            if(m_commands_socket.is_open())
                m_commands_socket.close(error);
            
            m_commands_socket_io_service.stop();

            m_data_socket_io_service_thread.join();
            
            if(m_log_level > 0) std::cout << "[II] lib3dv: destructor complete" << std::endl;
        }
        
        uint64_t device_impl::connect_image_callback(const boost::function<void (boost::shared_ptr<const lib3dv::image>, uint32_t)>& function)
        {            
            m_connections[m_max_connection_id] = m_image_signal.connect(function);
            ++m_max_connection_id;
            return (m_max_connection_id - 1);
        }
        
        uint64_t device_impl::connect_terrain_callback(const boost::function<void (boost::shared_ptr<const lib3dv::terrain>, uint32_t)>& function)
        {
            m_connections[m_max_connection_id] = m_terrain_signal.connect(function);
            ++m_max_connection_id;
            return (m_max_connection_id - 1);
        }
        
        uint64_t device_impl::connect_obstacles_callback(const boost::function<void (boost::shared_ptr<const std::vector<lib3dv::obstacle> >, uint32_t)>& function)
        {
            m_connections[m_max_connection_id] = m_obstacles_signal.connect(function);
            ++m_max_connection_id;
            return (m_max_connection_id - 1);
        }
        
        uint64_t device_impl::connect_motion_callback(const boost::function<void (boost::shared_ptr<const lib3dv::motion>, uint32_t)>& function)
        {
            m_connections[m_max_connection_id] = m_motion_signal.connect(function);
            ++m_max_connection_id;
            return (m_max_connection_id - 1);
        }

        uint64_t device_impl::connect_classification_callback(const boost::function<void (boost::shared_ptr<const lib3dv::classification>, uint32_t)>& function)
        {
            m_connections[m_max_connection_id] = m_classification_signal.connect(function);
            ++m_max_connection_id;
            return (m_max_connection_id - 1);
        }


        LIB3DV_DEPRECATED uint64_t device_impl::connect_image_callback(const boost::function<void (boost::shared_ptr<lib3dv::image>, uint32_t)>& function)
        {
        	m_connections[m_max_connection_id] = m_deprecated_image_signal.connect(function);
        	++m_max_connection_id;
        	return (m_max_connection_id - 1);
        }

        LIB3DV_DEPRECATED uint64_t device_impl::connect_terrain_callback(const boost::function<void (boost::shared_ptr<lib3dv::terrain>, uint32_t)>& function)
        {
        	m_connections[m_max_connection_id] = m_deprecated_terrain_signal.connect(function);
        	++m_max_connection_id;
        	return (m_max_connection_id - 1);
        }

        LIB3DV_DEPRECATED uint64_t device_impl::connect_obstacles_callback(const boost::function<void (boost::shared_ptr<std::vector<lib3dv::obstacle> >, uint32_t)>& function)
        {
        	m_connections[m_max_connection_id] = m_deprecated_obstacles_signal.connect(function);
        	++m_max_connection_id;
        	return (m_max_connection_id - 1);
        }

        LIB3DV_DEPRECATED uint64_t device_impl::connect_motion_callback(const boost::function<void (boost::shared_ptr<lib3dv::motion>, uint32_t)>& function)
        {
        	m_connections[m_max_connection_id] = m_deprecated_motion_signal.connect(function);
        	++m_max_connection_id;
        	return (m_max_connection_id - 1);
        }

        LIB3DV_DEPRECATED uint64_t device_impl::connect_classification_callback(const boost::function<void (boost::shared_ptr<lib3dv::classification>, uint32_t)>& function)
        {
        	m_connections[m_max_connection_id] = m_deprecated_classification_signal.connect(function);
        	++m_max_connection_id;
        	return (m_max_connection_id - 1);
        }

        uint64_t device_impl::connect_timeout_callback(const boost::function<void (void)>& function)
        {
            m_connections[m_max_connection_id] = m_timeout_signal.connect(function);
            ++m_max_connection_id;
            return (m_max_connection_id - 1);
        }
        
        void device_impl::disconnect_callback(uint64_t id)
        {
            std::map<uint64_t, boost::signals2::connection>::iterator cc = m_connections.find(id);
            
            if(cc != m_connections.end())
            {
                cc->second.disconnect();
                m_connections.erase(cc);
            }
        }
        

        
        void on_device_info_received(const boost::system::error_code& error,
                                     size_t bytes_received,
                                     boost::asio::ip::udp::socket& socket,
                                     boost::array<uint8_t, device_impl::MAX_UDP_PAYLOAD_SIZE>& receive_buffer,
                                     boost::asio::ip::udp::endpoint& local_endpoint,
                                     boost::asio::ip::udp::endpoint& remote_endpoint,
                                     std::vector<device>& devices,
                                     uint8_t log_level,
                                     lib3dv::error& device_error)
        {            
            if(!error)
            {
                if(bytes_received >= sizeof(protocol::packet_header) + sizeof(protocol::command_header))
                {
                    if(!memcmp(receive_buffer.data(), "#3DV", 4) && receive_buffer.data()[4] <= protocol::version::MAX_SUPPORTED)
                    {
                        const protocol::packet_header* packet_header = reinterpret_cast<const protocol::packet_header*>(receive_buffer.data());
                        const protocol::command_header* command_header = reinterpret_cast<const protocol::command_header*>(receive_buffer.data() + sizeof(protocol::packet_header));
                        const protocol::device_info* device_info = reinterpret_cast<const protocol::device_info*>(receive_buffer.data() + sizeof(protocol::packet_header) + sizeof(protocol::command_header));

                        device_error = static_cast<lib3dv::error::types>(packet_header->m_error);
                        
                        if(device_error == lib3dv::error::NONE)
                        {
                            boost::uuids::uuid guid;

                            std::copy(device_info->m_guid, device_info->m_guid + 16, guid.data);
                            
                            std::bitset<device::capability::NUM> capabilities;
                            
                            for(unsigned int i = 0; i < device::capability::NUM; ++i)
                                if((device_info->m_capabilities) & (1 << i))
                                    capabilities.set(i);
                            
                            device::version_info version;
                            const uint8_t* raw_framework_version = device_info->m_framework_version;
                            const uint8_t* raw_application_version = device_info->m_application_version;
                            
                            version.m_protocol = packet_header->m_protocol_version;
                            std::copy(device_info->m_framework_version, raw_framework_version + 3, version.m_framework);
                            std::copy(raw_application_version, raw_application_version + 3, version.m_application);
                                
                            device_impl* device = new device_impl(local_endpoint.address().to_v4(),
                                                                  remote_endpoint.address().to_v4(), remote_endpoint.port(), 0,
                                                                  guid,
                                                                  capabilities,
                                                                  version,
                                                                  log_level, device_error);

                            if(device_error == lib3dv::error::NONE)                            
                                devices.push_back(create(boost::shared_ptr<device_impl>(device)));
                            else
                                delete device;
                        }
                    }
                    else
                        std::cerr << "[EE] lib3dv: unknown packet received" << std::endl;
                }
                else
                {
                    std::cerr << "[EE] lib3dv: " << error.message() << std::endl;
                    device_error = lib3dv::error::NETWORK_FAILURE;
                }
                
                if(socket.is_open())
                    socket.async_receive_from(boost::asio::buffer(receive_buffer), remote_endpoint, boost::bind(on_device_info_received, _1, _2,
                                                                                                                boost::ref(socket), boost::ref(receive_buffer),
                                                                                                                boost::ref(local_endpoint), boost::ref(remote_endpoint),
                                                                                                                boost::ref(devices), log_level, boost::ref(device_error)));
            }
        }
        
        void cleanup(boost::asio::io_service& io_service, boost::asio::ip::udp::socket& socket, lib3dv::error& device_error)
        {
            boost::system::error_code error;
            
            if(socket.is_open())
                socket.close(error);
            
            device_error = error ? lib3dv::error::NETWORK_FAILURE : lib3dv::error::NONE;
            
            io_service.stop();
        }
        
        std::vector<device> device_impl::enumerate(const boost::asio::ip::address_v4& local_address, unsigned short remote_port,
                                                                       uint8_t log_level,
                                                                       lib3dv::error& device_error,
                                                                       const boost::posix_time::time_duration& timeout)
        {
            std::vector<device> devices;
            boost::asio::ip::udp::endpoint local_endpoint(local_address, 0);
            
            boost::asio::io_service io_service;
            boost::asio::ip::udp::socket socket(io_service);
            boost::asio::deadline_timer deadline(io_service);
            boost::system::error_code error;
                     
            socket.open(boost::asio::ip::udp::v4(), error);
            
            if(error)
            {
                std::cerr << "[EE] lib3dv: " << error.message() << std::endl;                
                device_error = lib3dv::error::NETWORK_FAILURE;
                return devices;
            }
            
            socket.set_option(boost::asio::socket_base::broadcast(true));
            socket.bind(local_endpoint, error);
            
            if(error)
            {
                std::cerr << "[EE] lib3dv: " << error.message() << std::endl;
                device_error = lib3dv::error::NETWORK_FAILURE;
                return devices;
            }
            
            boost::asio::ip::udp::endpoint remote_endpoint;
            boost::array<uint8_t, MAX_UDP_PAYLOAD_SIZE> receive_buffer;
                        
            socket.async_receive_from(boost::asio::buffer(receive_buffer), remote_endpoint, boost::bind(on_device_info_received, _1, _2,
                                                                                                        boost::ref(socket), boost::ref(receive_buffer),
                                                                                                        boost::ref(local_endpoint), boost::ref(remote_endpoint),
                                                                                                        boost::ref(devices), log_level, boost::ref(device_error)));

            deadline.expires_from_now(timeout);
            on_deadline(deadline, socket, log_level, false, io_service, boost::bind(cleanup, boost::ref(io_service), boost::ref(socket), boost::ref(device_error)), boost::system::error_code());

            boost::thread io_service_thread(boost::bind(&boost::asio::io_service::run, &io_service));
            
            boost::asio::ip::address_v4 remote_broadcast_address = boost::asio::ip::address_v4::broadcast(local_address, boost::asio::ip::address_v4::netmask(local_address));

            boost::shared_ptr<protocol::packet_header> packet_header = boost::shared_ptr<protocol::packet_header>(new protocol::packet_header());
            boost::shared_ptr<protocol::command_header> raw_command = boost::shared_ptr<protocol::command_header>(new protocol::command_header());
            boost::shared_ptr<std::vector<uint8_t> > raw_command_data = boost::shared_ptr<std::vector<uint8_t> >(new std::vector<uint8_t>());            
            
            packet_header->m_magic[0] = '#';
            packet_header->m_magic[1] = '3';
            packet_header->m_magic[2] = 'D';
            packet_header->m_magic[3] = 'V';
            packet_header->m_protocol_version = 0;
            packet_header->m_fragment = 0;
            packet_header->m_total_fragments = 1;
            packet_header->m_payload_type = protocol::payload::COMMAND;
            
            raw_command->m_type = lib3dv::detail::command::ENUMERATE_DEVICES;
            raw_command->m_size = 0;
            
            boost::array<boost::asio::const_buffer, 3> buffers;
            
            buffers[0] = boost::asio::buffer(packet_header.get(), sizeof(protocol::packet_header));
            buffers[1] = boost::asio::buffer(raw_command.get(), sizeof(protocol::command_header));
            buffers[2] = boost::asio::buffer(raw_command_data->data(), raw_command_data->size());

            if(log_level > 0) std::cout << "[II] lib3dv: broadcasting discovery message to " << remote_broadcast_address << ':' << remote_port << std::endl;
            
            socket.send_to(buffers, boost::asio::ip::udp::endpoint(remote_broadcast_address, remote_port), 0, error);
            
            if(error)
            {
                std::cerr << "[EE] lib3dv: " << error.message() << std::endl;                
                device_error = lib3dv::error::NETWORK_FAILURE;
                return devices;
            }
            
            if(log_level > 0) std::cout << "[II] lib3dv: broadcast complete" << std::endl;            
            
            io_service_thread.join();
            
            if(log_level > 0) std::cout << "[II] lib3dv: device discovery complete" << std::endl;

            device_error = lib3dv::error::NONE;
            
            return devices;
        }
        
        void device_impl::on_data_received(const boost::system::error_code& error, size_t bytes_received)
        {
            if(m_log_level > 1) std::cout << std::endl << "[II] lib3dv: " << " " << bytes_received << " data bytes read" << std::endl;

            m_data_deadline.expires_from_now(m_timeout);
            
            if(bytes_received >= sizeof(protocol::packet_header) && !memcmp(m_data_payload.c_array(), "#3DV", 4))
            {
                protocol::packet_header* packet_header = reinterpret_cast<protocol::packet_header*>(m_data_payload.c_array());
                
                if(m_log_level > 1)
                {
                    std::cout << "\t" << "protocol version = " << (int)(packet_header->m_protocol_version) << std::endl;
                    std::cout << "\t" << "guid = " << packet_header->m_guid << std::endl;
                    std::cout << "\t" << "fragment = " << packet_header->m_fragment << " / " << packet_header->m_total_fragments << std::endl;
                    std::cout << "\t" << "payload_type = " << (int)packet_header->m_payload_type << std::endl;
                }

                cleanup_reassembly_container(m_images, m_image_fragments);
                cleanup_reassembly_container(m_terrains, m_terrain_fragments);
                cleanup_reassembly_container(m_obstacles, m_obstacle_fragments);
                cleanup_reassembly_container(m_poses, m_pose_fragments);
                
                switch(packet_header->m_payload_type)
                {                    
                    case protocol::payload::IMAGE_INFO_HEADER :
                    {
                        protocol::image_info_header* image_info_header = reinterpret_cast<protocol::image_info_header*>(m_data_payload.c_array() + sizeof(protocol::packet_header));

                        if(m_log_level > 0 && image_info_header->m_type)
                        {
                            std::cout << (m_log_level == 1 ? "\n" : "");
                            std::cout << (m_log_level > 1 ? "\t" : "") << "image" << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "  guid = " << packet_header->m_guid << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "  header:" << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "    bpp = " << (int)image_info_header->m_bpp << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "    channels = " << (int)image_info_header->m_channels << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "    width = " << image_info_header->m_width << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "    height = " << image_info_header->m_height << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "    size = " << image_info_header->m_size << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "    type = " << (int)image_info_header->m_type << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "    format = " << (int)image_info_header->m_format << std::endl;
                            std::cout << (m_log_level == 1 ? "  data: " : "");
                        }
                        
                        boost::shared_ptr<raw_image_type> image = init_reassembly_container(m_images, m_image_fragments, *image_info_header, packet_header->m_guid);

                        if(image)
                        {
                            boost::shared_ptr<lib3dv::image> output_image = boost::shared_ptr<lib3dv::image>(new lib3dv::image());

                            output_image->m_width = image->m_header.m_width;
                            output_image->m_height = image->m_header.m_height;
                            output_image->m_channels = image->m_header.m_channels;
                            output_image->m_bpp = image->m_header.m_bpp;
                            output_image->m_type = static_cast<lib3dv::image::type::types>(image->m_header.m_type);
                            output_image->m_format = static_cast<lib3dv::image::format::types>(image->m_header.m_format);
                            output_image->m_timestamp = boost::posix_time::ptime(boost::gregorian::date(1970,1,1)) +
                                                        boost::posix_time::seconds(image->m_header.m_timestamp / 1000000) + // this serves as a workaround to an integer overflow in boost versions <= 1.51
                                                        boost::posix_time::microseconds(image->m_header.m_timestamp % 1000000);
                            output_image->m_buffer.swap(image->m_buffer);

                            m_image_signal(output_image, packet_header->m_guid);
                            m_deprecated_image_signal(output_image, packet_header->m_guid);
                        }

                        break;
                    }

                    case protocol::payload::IMAGE_DATA :
                    {
                        boost::shared_ptr<raw_image_type> image = fill_reassembly_container(m_images, m_image_fragments,
                                                                                            bytes_received, packet_header->m_guid,
                                                                                            packet_header->m_fragment, packet_header->m_total_fragments);

                        if(image)
                        {
                            boost::shared_ptr<lib3dv::image> output_image = boost::shared_ptr<lib3dv::image>(new lib3dv::image());
                            
                            output_image->m_width = image->m_header.m_width;
                            output_image->m_height = image->m_header.m_height;
                            output_image->m_channels = image->m_header.m_channels;
                            output_image->m_bpp = image->m_header.m_bpp;
                            output_image->m_type = static_cast<lib3dv::image::type::types>(image->m_header.m_type);
                            output_image->m_format = static_cast<lib3dv::image::format::types>(image->m_header.m_format);
                            output_image->m_timestamp = boost::posix_time::ptime(boost::gregorian::date(1970,1,1)) +
                                                        boost::posix_time::seconds(image->m_header.m_timestamp / 1000000) + // this serves as a workaround to an integer overflow in boost versions <= 1.51
                                                        boost::posix_time::microseconds(image->m_header.m_timestamp % 1000000);
                            output_image->m_buffer.swap(image->m_buffer);

                            m_image_signal(output_image, packet_header->m_guid);
                            m_deprecated_image_signal(output_image, packet_header->m_guid);
                        }

                        break;
                    }
                    
                    case protocol::payload::TERRAIN_INFO_HEADER :
                    {
                        protocol::terrain_info_header* terrain_info_header = reinterpret_cast<protocol::terrain_info_header*>(m_data_payload.c_array() + sizeof(protocol::packet_header));

                        if(m_log_level > 0)
                        {
                            std::cout << (m_log_level == 1 ? "\n" : "");
                            std::cout << (m_log_level > 1 ? "\t" : "") << "terrain" << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "  guid = " << packet_header->m_guid << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "  header:" << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "    x cells number = " << (int)terrain_info_header->m_x_cell_num << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "    x cells size = " << (int)terrain_info_header->m_x_cell_size << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "    x min = " << terrain_info_header->m_x_min << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "    y cells number = " << (int)terrain_info_header->m_y_cell_num << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "    y cells size = " << (int)terrain_info_header->m_y_cell_size << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "    y min = " << terrain_info_header->m_y_min << std::endl;
                            std::cout << (m_log_level == 1 ? "  data: " : "");
                        }
                        
                        boost::shared_ptr<raw_terrain_type> terrain = init_reassembly_container(m_terrains, m_terrain_fragments, *terrain_info_header, packet_header->m_guid);

                        if(terrain)
                        	decode_terrain(terrain, packet_header->m_guid);

                        break;
                    }
                    
                    case protocol::payload::TERRAIN_DATA :
                    {
                        boost::shared_ptr<raw_terrain_type> terrain = fill_reassembly_container(m_terrains, m_terrain_fragments,
                                                                                                bytes_received, packet_header->m_guid,
                                                                                                packet_header->m_fragment, packet_header->m_total_fragments);

                        if(terrain)
                            decode_terrain(terrain, packet_header->m_guid);
                        
                        break;
                    }
                    
                    case protocol::payload::OBSTACLES_MAP_INFO_HEADER :
                    {
                        protocol::obstacles_map_info_header* obstacles_map_info_header = reinterpret_cast<protocol::obstacles_map_info_header*>(m_data_payload.c_array() + sizeof(protocol::packet_header));

                        if(m_log_level > 0)
                        {
                            std::cout << (m_log_level == 1 ? "\n" : "");
                            std::cout << (m_log_level > 1 ? "\t" : "") << "obstacles" << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "  guid = " << packet_header->m_guid << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "  header:" << std::endl;
                            std::cout << (m_log_level > 1 ? "\t" : "") << "    obstacles number = " << (int)obstacles_map_info_header->m_obstacles_num << std::endl;
                            std::cout << (m_log_level == 1 ? "  data: " : "");
                        }
                        
                        boost::shared_ptr<raw_obstacles_type> obstacles = init_reassembly_container(m_obstacles, m_obstacle_fragments, *obstacles_map_info_header, packet_header->m_guid);

                        if(obstacles)
							decode_obstacles(obstacles, packet_header->m_guid);

                        break;
                    }
                    
                    case protocol::payload::OBSTACLES_MAP_DATA:
                    {
                        boost::shared_ptr<raw_obstacles_type> obstacles = fill_reassembly_container(m_obstacles, m_obstacle_fragments,
                                                                                                bytes_received, packet_header->m_guid,
                                                                                                packet_header->m_fragment, packet_header->m_total_fragments);
                        if(obstacles)
                            decode_obstacles(obstacles, packet_header->m_guid);
                        
                        break;
                    }
                    
                    case protocol::payload::MOTION_INFO_HEADER :
                    {
                    	protocol::motion_info_header* motion_info_header = reinterpret_cast<protocol::motion_info_header*>(m_data_payload.c_array() + sizeof(protocol::packet_header));

                    	if(m_log_level > 0)
                    	{
                    		std::cout << (m_log_level == 1 ? "\n" : "");
                    		std::cout << (m_log_level > 1 ? "\t" : "") << "pose" << std::endl;
                    		std::cout << (m_log_level > 1 ? "\t" : "") << "  guid = " << packet_header->m_guid << std::endl;
                    		std::cout << (m_log_level > 1 ? "\t" : "") << "  header:" << std::endl;
                    		std::cout << (m_log_level > 1 ? "\t" : "") << "      size = " << motion_info_header->m_size << std::endl;
                    		std::cout << (m_log_level > 1 ? "\t" : "") << "      poses  = " << (int) motion_info_header->m_poses_num << std::endl;
                    		std::cout << (m_log_level == 1 ? "  data: " : "");
                    	}

                    	boost::shared_ptr<raw_motion_type> motion = init_reassembly_container(m_poses, m_pose_fragments, *motion_info_header, packet_header->m_guid);

                    	if(motion)
							decode_motion(motion, packet_header->m_guid);

                    	break;
                    }

                    case protocol::payload::MOTION_DATA :
                    {
                    	boost::shared_ptr<raw_motion_type> motion = fill_reassembly_container(m_poses, m_pose_fragments,
                    			bytes_received, packet_header->m_guid,
                    			packet_header->m_fragment, packet_header->m_total_fragments);

                    	if(motion)
                    		decode_motion(motion, packet_header->m_guid);

                    	break;
                    }

                    case protocol::payload::CLASSIFICATION_INFO_HEADER :
                    {
                    	//std::cout << "classification header!"<<std::endl;

                    	protocol::classifier_info_header* classifier_info_header = reinterpret_cast<protocol::classifier_info_header*>(m_data_payload.c_array() + sizeof(protocol::packet_header));

                    	if(m_log_level > 0)
                    	{
                    		std::cout << (m_log_level == 1 ? "\n" : "");
                    		std::cout << (m_log_level > 1 ? "\t" : "") << "candidate" << std::endl;
                    		std::cout << (m_log_level > 1 ? "\t" : "") << "  guid = " << packet_header->m_guid << std::endl;
                    		std::cout << (m_log_level > 1 ? "\t" : "") << "  header:" << std::endl;
                    		std::cout << (m_log_level > 1 ? "\t" : "") << "      size = " << classifier_info_header->m_size << std::endl;
                    		std::cout << (m_log_level == 1 ? "  data: " : "");
                    	}

                    	boost::shared_ptr<raw_candidate_type> classification = init_reassembly_container(m_candidates, m_candidates_fragments, *classifier_info_header, packet_header->m_guid);

                    	if(classification)
							decode_classification(classification, packet_header->m_guid);

                    	break;
                    }

                    case protocol::payload::CLASSIFICATION_DATA :
                    {
                    	//std::cout << "classification data!"<<std::endl;

                    	boost::shared_ptr<raw_candidate_type> classification = fill_reassembly_container(m_candidates, m_candidates_fragments,
                    			bytes_received, packet_header->m_guid,
                    			packet_header->m_fragment, packet_header->m_total_fragments);

                    	if(classification)
                    		decode_classification(classification, packet_header->m_guid);

                    	break;
                    }


                    case protocol::payload::KEEP_ALIVE:
                    {
                        break;
                    }

                    default:
                        std::cerr << "[EE] lib3dv: unknown payload type: " << (int)packet_header->m_payload_type << std::endl;
                }
            }

            m_data_socket.async_receive_from(boost::asio::buffer(&m_data_payload, MAX_UDP_PAYLOAD_SIZE),
                                        m_data_sender_endpoint,
                                        boost::bind(&device_impl::on_data_received, this, boost::asio::placeholders::error, boost::asio::placeholders::bytes_transferred));
        }
        
        void device_impl::decode_terrain(boost::shared_ptr<raw_terrain_type> terrain, uint32_t guid)
        {
            boost::shared_ptr<lib3dv::terrain> output_terrain = boost::shared_ptr<lib3dv::terrain>(new lib3dv::terrain());
            output_terrain->m_data.reserve(terrain->m_buffer.size() / 2);

            float x_min = terrain->m_header.m_x_min / 100.0f;
            float y_min = terrain->m_header.m_y_min / 100.0f;
            
            float x_cell_size = terrain->m_header.m_x_cell_size / 100.0f;
            float y_cell_size = terrain->m_header.m_y_cell_size / 100.0f;
            
            uint16_t x_cell_num = terrain->m_header.m_x_cell_num;
            uint16_t y_cell_num = terrain->m_header.m_y_cell_num;                                
            
            const int16_t* data = reinterpret_cast<const int16_t*>(terrain->m_buffer.data());
            std::vector<lib3dv::point3>& output_terrain_data = output_terrain->m_data;

            lib3dv::point3 p;
            unsigned int t = 0;
            
            for(unsigned int x = 0; x < x_cell_num; ++x)
            {                                    
                p.m_x = x_min + x * x_cell_size;
                
                for(unsigned int y = 0; y < y_cell_num; ++y)
                {
                    if(data[t] != static_cast<int16_t>(0x8000))
                    {
                        p.m_y = y_min + y * y_cell_size;
                        p.m_z = data[t] / 100.0f;
                        output_terrain_data.push_back(p);
                    }
                    
                    ++t;
                }
            }

            m_terrain_signal(output_terrain, guid);
            m_deprecated_terrain_signal(output_terrain, guid);
        }
        
        void device_impl::decode_obstacles(boost::shared_ptr<raw_obstacles_type> obstacles, uint32_t guid)
        {
            boost::shared_ptr<std::vector<lib3dv::obstacle> > output_obstacles = boost::shared_ptr<std::vector<lib3dv::obstacle> >(new std::vector<lib3dv::obstacle>(obstacles->m_header.m_obstacles_num));            
            uint8_t* raw_obstacle = obstacles->m_buffer.data();
            
            for(std::vector<lib3dv::obstacle>::iterator oo = output_obstacles->begin(); oo != output_obstacles->end(); ++oo)
            {
                const protocol::obstacle_info_header* obstacle_header = reinterpret_cast<const protocol::obstacle_info_header*>(raw_obstacle);
                const protocol::stixel* obstacle_data = reinterpret_cast<const protocol::stixel*>(raw_obstacle + sizeof(protocol::obstacle_info_header));
                
                oo->m_guid = obstacle_header->m_guid;
                oo->m_data.resize(obstacle_header->m_stixel_num);
                                
                for(unsigned int s = 0; s < obstacle_header->m_stixel_num; ++s)
                {
                    lib3dv::stixel& stixel = oo->m_data[s];

                    stixel.m_dx = obstacle_data[s].m_dx / 100.0f;
                    stixel.m_dy = obstacle_data[s].m_dy / 100.0f;
                    stixel.m_x = obstacle_data[s].m_x / 100.0f;
                    stixel.m_y = obstacle_data[s].m_y / 100.0f;
                    stixel.m_z = obstacle_data[s].m_z / 100.0f;
                    stixel.m_height = obstacle_data[s].m_height / 100.0f;
                }
                
                raw_obstacle += (sizeof(protocol::obstacle_info_header) + obstacle_header->m_stixel_num * sizeof(protocol::stixel));
            }
            
            m_obstacles_signal(output_obstacles, guid);
            m_deprecated_obstacles_signal(output_obstacles, guid);
        }
        
        void device_impl::decode_motion(boost::shared_ptr<raw_motion_type> motion, uint32_t guid)
        {
        	boost::shared_ptr<lib3dv::motion> output_motion = boost::shared_ptr<lib3dv::motion>(new lib3dv::motion());
        	output_motion->m_poses.resize(motion->m_header.m_poses_num);

        	std::vector<lib3dv::pose>& output_poses = output_motion->m_poses;
        	uint8_t* raw_pose = motion->m_buffer.data();

        	for(std::vector<lib3dv::pose>::iterator pose = output_poses.begin(); pose != output_poses.end(); ++pose)
        	{
        		const protocol::pose_info_header* pose_header = reinterpret_cast<const protocol::pose_info_header*>(raw_pose);
        		raw_pose +=  sizeof(protocol::pose_info_header);

        		const double* pose_data = reinterpret_cast<const double*>(raw_pose);

        		pose->m_timestamp = boost::posix_time::ptime(boost::gregorian::date(1970,1,1)) +
        				boost::posix_time::seconds(pose_header->m_timestamp / 1000000) + // this serves as a workaround to an integer overflow in boost versions <= 1.51
        				boost::posix_time::microseconds(pose_header->m_timestamp % 1000000);

        		if ( pose_header->m_type == 0 )
        			pose->m_type = lib3dv::pose::type::CURRENT_POSE;
        		else if ( pose_header->m_type == 1 )
        			pose->m_type = lib3dv::pose::type::RELATIVE_POSE;


        		pose->m_data.resize(pose_header->m_pose_num);

        		for (uint8_t i = 0; i < pose->m_data.size(); i++)
        			pose->m_data[i] = pose_data[i];

        		raw_pose += (pose_header->m_pose_num * sizeof(double));
        	}

        	m_motion_signal(output_motion, guid);
        	m_deprecated_motion_signal(output_motion, guid);
        }

        void device_impl::decode_classification(boost::shared_ptr<raw_candidate_type> candidates, uint32_t guid)
        {
            boost::shared_ptr<lib3dv::classification> output_classification = boost::shared_ptr<lib3dv::classification>(new lib3dv::classification());
            output_classification->m_candidates.resize(candidates->m_header.m_candidates_num);
            uint8_t* raw_candidate = candidates->m_buffer.data();
            output_classification->m_timestamp = boost::posix_time::ptime(boost::gregorian::date(1970,1,1)) +
    				boost::posix_time::seconds(candidates->m_header.m_timestamp / 1000000) + // this serves as a workaround to an integer overflow in boost versions <= 1.51
    				boost::posix_time::microseconds(candidates->m_header.m_timestamp % 1000000);
            for(std::vector<lib3dv::candidate>::iterator oo = output_classification->m_candidates.begin(); oo != output_classification->m_candidates.end(); ++oo)
            {
                //const protocol::classifier_info_header * classifier_header = reinterpret_cast<const protocol::classifier_info_header*>(raw_candidate);
                //const protocol::classifier_data* candidate_data = reinterpret_cast<const protocol::classifier_data*>(raw_candidate + sizeof(protocol::classifier_info_header));
            	const protocol::classifier_data* candidate_data = reinterpret_cast<const protocol::classifier_data*>(raw_candidate);
                //oo->m_category = lib3dv::category_type::types (candidate_data->m_category);
                //oo->m_confidence = candidate_data->m_confidence;
                //oo->m_wp_lb.m_x = candidate_data->m_wp_lb_x;
                //oo->m_wp_lb.m_y = candidate_data->m_wp_lb_y;
                //oo->m_wp_lb.m_z = candidate_data->m_wp_lb_z;
                oo->m_x0 = candidate_data->m_x0;
                oo->m_y0 = candidate_data->m_y0;
                oo->m_x1 = candidate_data->m_x1;
                oo->m_y1 = candidate_data->m_y1;
                oo->m_confidence = candidate_data->m_confidence/100.0;
                oo->m_category = lib3dv::category_type::types (candidate_data->m_category);
                oo->m_guid = candidate_data->m_guid;

                //std::cout << oo->m_guid <<"("<<"\t"<< oo->m_x0<<","<<oo->m_y0<<","<<oo->m_x1<<","<<oo->m_y1 << ") "<<oo->m_confidence<< " "<< oo->m_category<<std::endl;
                raw_candidate += sizeof(protocol::classifier_data);
            }
            boost::shared_ptr<lib3dv::classification> output_motion = boost::shared_ptr<lib3dv::classification>(new lib3dv::classification());
            m_classification_signal(output_classification, guid);
            m_deprecated_classification_signal(output_classification, guid);
        }


        void device_impl::save_properties(error& error)
        {
            boost::shared_ptr<detail::command> command = boost::shared_ptr<detail::command>(new detail::command());
            
            command->m_type = detail::command::SAVE_PROPERTIES;
                        
            send_command(command, error);

            m_commands_socket_io_service.run();
            m_commands_socket_io_service.reset();
        }

        void device_impl::reset_properties(error& error)
        {
            boost::shared_ptr<detail::command> command = boost::shared_ptr<detail::command>(new detail::command());
            
            command->m_type = detail::command::RESET_PROPERTIES;
                        
            send_command(command, error);

            m_commands_socket_io_service.run();
            m_commands_socket_io_service.reset();
        }
        
        std::vector<device::property> device_impl::enumerate_properties(lib3dv::error& error)
        {
            boost::shared_ptr<detail::command> command = boost::shared_ptr<detail::command>(new detail::command());
            std::vector<device::property> properties;
            
            command->m_type = detail::command::ENUMERATE_PROPERTIES;
            command->m_result = boost::ref(properties);

            send_command(command, error);

            m_commands_socket_io_service.run();
            m_commands_socket_io_service.reset();
            
            return properties;
        }    
        
        boost::any device_impl::get_property_value(uint16_t address, lib3dv::error& error)
        {
            boost::shared_ptr<detail::command> command = boost::shared_ptr<detail::command>(new detail::command());
            
            command->m_type = detail::command::GET_PROPERTY;
            command->m_data.resize(sizeof(uint16_t));
            *(reinterpret_cast<uint16_t*>(command->m_data.data())) = address;
            
            send_command(command, error);
            
            m_commands_socket_io_service.run();
            m_commands_socket_io_service.reset();

            return command->m_result;
        }
         
        template<typename T>
        size_t set_property_value_helper(uint8_t type, size_t size, protocol::param_info& info, boost::shared_ptr<detail::command> command, const boost::any& value)
        {
            info.m_value_type = type;                
            size += sizeof(T);
            command->m_data.resize(size);                

            *(reinterpret_cast<T*>(command->m_data.data() + sizeof(protocol::param_info))) = boost::any_cast<T>(value);
            
            return size;
        }

        template<>
        size_t set_property_value_helper<bool>(uint8_t type, size_t size, protocol::param_info& info, boost::shared_ptr<detail::command> command, const boost::any& value)
        {
            info.m_value_type = type;
            size += sizeof(uint8_t);
            command->m_data.resize(size);                

            *(reinterpret_cast<uint8_t*>(command->m_data.data() + sizeof(protocol::param_info))) = static_cast<uint8_t>(boost::any_cast<bool>(value));
            
            return size;
        }
        
        template<>
        size_t set_property_value_helper<std::string>(uint8_t type, size_t size, protocol::param_info& info, boost::shared_ptr<detail::command> command, const boost::any& value)
        {
            const std::string& str_value = boost::any_cast<std::string>(value);
            size_t str_size = std::min(str_value.size(), (size_t)255);
            
            info.m_value_type = type;
            size += (sizeof(uint8_t) + str_size);
            command->m_data.resize(size);
            
            uint8_t* data = command->m_data.data() + sizeof(protocol::param_info);
            
            *(reinterpret_cast<uint8_t*>(data)) = str_size;
            data += sizeof(uint8_t);

            std::copy(str_value.data(), str_value.data() + str_size, data);
            data += str_size;
            
            return size;
        }
        
        void device_impl::set_property_value(uint16_t address, const boost::any& value, lib3dv::error& error)
        {
            boost::shared_ptr<detail::command> command = boost::shared_ptr<detail::command>(new detail::command());
            
            size_t size = sizeof(protocol::param_info);
            
            protocol::param_info info;
            
            info.m_value_address = address;
            info.m_readonly = 0;
            info.m_category = 0;
            info.m_value_description_size = 0;
            
            const std::type_info& type = value.type();
            
            if(type == typeid(bool)) size = set_property_value_helper<bool>(command::BOOL, size, info, command, value);
            else if(type == typeid(int64_t)) size = set_property_value_helper<int64_t>(command::INT64, size, info, command, value);
            else if(type == typeid(int32_t)) size = set_property_value_helper<int32_t>(command::INT32, size, info, command, value);
            else if(type == typeid(int16_t)) size = set_property_value_helper<int16_t>(command::INT16, size, info, command, value);
            else if(type == typeid(int8_t)) size = set_property_value_helper<int8_t>(command::INT8, size, info, command, value);
            else if(type == typeid(uint64_t)) size = set_property_value_helper<uint64_t>(command::UINT64, size, info, command, value);
            else if(type == typeid(uint32_t)) size = set_property_value_helper<uint32_t>(command::UINT32, size, info, command, value);
            else if(type == typeid(uint16_t)) size = set_property_value_helper<uint16_t>(command::UINT16, size, info, command, value);
            else if(type == typeid(uint8_t)) size = set_property_value_helper<uint8_t>(command::UINT8, size, info, command, value);
            else if(type == typeid(float)) size = set_property_value_helper<float>(command::FLOAT32, size, info, command, value);
            else if(type == typeid(double)) size = set_property_value_helper<double>(command::FLOAT64, size, info, command, value);
            else if(type == typeid(std::string)) size = set_property_value_helper<std::string>(command::STRING, size, info, command, value);
            
            command->m_type = detail::command::SET_PROPERTY;
            *(reinterpret_cast<protocol::param_info*>(command->m_data.data())) = info;
            
            send_command(command, error);
            
            m_commands_socket_io_service.run();
            m_commands_socket_io_service.reset();
        }
        
        void device_impl::start_transmission(lib3dv::error& error)
        {
            boost::shared_ptr<detail::command> command = boost::shared_ptr<detail::command>(new detail::command());
        
            command->m_type = detail::command::START_TRANSMISSION;
            command->m_data.resize(sizeof(uint16_t));
            *(reinterpret_cast<uint16_t*>(command->m_data.data())) = m_data_socket.local_endpoint().port();
            
            send_command(command, error);

            m_data_deadline.expires_from_now(m_timeout);

            on_deadline(m_data_deadline, m_data_socket, m_log_level, true, m_data_socket_io_service, boost::bind(&device_impl::on_timeout, this, boost::ref(error)), boost::system::error_code());
            
            m_commands_socket_io_service.run();
            m_commands_socket_io_service.reset();
            
            if(error != lib3dv::error::NONE)
                m_status = device::status::OFFLINE;
            else
                m_status = device::status::TRANSMITTING;
        }
        
        void device_impl::stop_transmission(lib3dv::error& error)
        {
            boost::shared_ptr<detail::command> command = boost::shared_ptr<detail::command>(new detail::command());
        
            command->m_type = detail::command::STOP_TRANSMISSION;
            
            send_command(command, error);

            m_commands_socket_io_service.run();
            m_commands_socket_io_service.reset();
            
            if(error != lib3dv::error::NONE)
                m_status = device::status::OFFLINE;
            else
                m_status = device::status::ONLINE;
        }
        
        void device_impl::poweroff(lib3dv::error& error)
        {
        	m_status = device::status::OFFLINE;

            boost::shared_ptr<detail::command> command = boost::shared_ptr<detail::command>(new detail::command());
        
            command->m_type = detail::command::POWEROFF;
            
            send_command(command, error);

            m_commands_socket_io_service.run();
            m_commands_socket_io_service.reset();
        }

        void device_impl::reboot(lib3dv::error& error)
        {
        	m_status = device::status::OFFLINE;

            boost::shared_ptr<detail::command> command = boost::shared_ptr<detail::command>(new detail::command());
        
            command->m_type = detail::command::REBOOT;
            
            send_command(command, error);

            m_commands_socket_io_service.run();
            m_commands_socket_io_service.reset();
        }
        
        void device_impl::send_command(boost::shared_ptr<lib3dv::detail::command> command, lib3dv::error& error)
        {
            error = lib3dv::error::NONE;

            boost::shared_ptr<protocol::packet_header> packet_header = boost::shared_ptr<protocol::packet_header>(new protocol::packet_header());
            boost::shared_ptr<protocol::command_header> raw_command = boost::shared_ptr<protocol::command_header>(new protocol::command_header());
            boost::shared_ptr<std::vector<uint8_t> > raw_command_data = boost::shared_ptr<std::vector<uint8_t> >(new std::vector<uint8_t>());            
            
            packet_header->m_magic[0] = '#';
            packet_header->m_magic[1] = '3';
            packet_header->m_magic[2] = 'D';
            packet_header->m_magic[3] = 'V';
            packet_header->m_protocol_version = 0;
            packet_header->m_fragment = 0;
            packet_header->m_total_fragments = 1;
            packet_header->m_payload_type = protocol::payload::COMMAND;
            
            raw_command->m_type = command->m_type;
            raw_command->m_size = command->m_data.size();

            raw_command_data->swap(command->m_data);
            
            boost::array<boost::asio::const_buffer, 3> buffers;
            
            buffers[0] = boost::asio::buffer(packet_header.get(), sizeof(protocol::packet_header));
            buffers[1] = boost::asio::buffer(raw_command.get(), sizeof(protocol::command_header));
            buffers[2] = boost::asio::buffer(raw_command_data->data(), raw_command_data->size());
            
            m_commands_socket.async_send_to(buffers,
                                            boost::asio::ip::udp::endpoint(m_remote_address, m_remote_commands_port),
                                            boost::bind(&device_impl::on_command_sent, this,
                                                        boost::asio::placeholders::error,
                                                        boost::asio::placeholders::bytes_transferred,
                                                        boost::ref(command->m_result), boost::ref(error)));
            
            m_commands_deadline.expires_from_now(m_timeout);
            on_deadline(m_commands_deadline, m_commands_socket, m_log_level, false, m_commands_socket_io_service, boost::bind(&device_impl::on_timeout, this, boost::ref(error)), boost::system::error_code());
        }
        
        void device_impl::on_timeout(lib3dv::error& device_error)
        {
            // we don't have to close the connections if the timeout has been triggered by a stop_transmission()
            if(m_status == device::status::TRANSMITTING)
            {
                boost::system::error_code error;
                
                if(m_commands_socket.is_open())
                    m_commands_socket.close(error);
                    
                m_commands_socket_io_service.stop();
                
                if(m_data_socket.is_open())
                    m_data_socket.close(error);
                
                m_data_socket_io_service.stop();

                m_status = device::status::OFFLINE;

                device_error = error ? lib3dv::error::NETWORK_FAILURE : lib3dv::error::NETWORK_TIMEOUT;
                
                m_timeout_signal();
            }
        }
        
        void device_impl::on_command_sent(const boost::system::error_code& error, size_t bytes_sent, boost::any& result, lib3dv::error& device_error)
        {
            if(m_log_level > 1) std::cout << "[II] lib3dv: command sent " << error << " (" << bytes_sent << " bytes)" << std::endl;
                        
            m_commands_socket.async_receive_from(boost::asio::buffer(m_commands_payload), m_commands_sender_endpoint, boost::bind(&device_impl::on_command_received, this,
                                                                                                                                    boost::asio::placeholders::error,
                                                                                                                                    boost::asio::placeholders::bytes_transferred,
                                                                                                                                    boost::ref(result), boost::ref(device_error)));
        }
        
        inline const uint8_t* decode_value(const uint8_t* value, command::values type, boost::any& result)
        {
            switch(type)
            {
                case command::BOOL : { result = static_cast<bool>(*reinterpret_cast<const uint8_t*>(value)); value += sizeof(uint8_t); break; }
                
                case command::INT64 : { result = *reinterpret_cast<const int64_t*>(value); value += sizeof(int64_t); break; }
                
                case command::INT32 : { result = *reinterpret_cast<const int32_t*>(value); value += sizeof(int32_t); break; }
                
                case command::INT16 : { result = *reinterpret_cast<const int16_t*>(value); value += sizeof(int16_t); break; }
                
                case command::INT8 : { result = *reinterpret_cast<const int8_t*>(value); value += sizeof(int8_t); break; }

                case command::UINT64 : { result = *reinterpret_cast<const uint64_t*>(value); value += sizeof(uint64_t); break; }
                
                case command::UINT32 : { result = *reinterpret_cast<const uint32_t*>(value); value += sizeof(uint32_t); break; }
                
                case command::UINT16 : { result = *reinterpret_cast<const uint16_t*>(value); value += sizeof(uint16_t); break; }
                
                case command::UINT8 : { result = *reinterpret_cast<const uint8_t*>(value); value += sizeof(uint8_t); break; }
                
                case command::FLOAT32 : { result = *reinterpret_cast<const float*>(value); value += sizeof(float); break; }
                
                case command::FLOAT64 : { result = *reinterpret_cast<const double*>(value); value += sizeof(double); break; }
                
                case command::STRING :
                {
                    size_t size = *value;
                    value += sizeof(uint8_t);
                    
                    result = std::string(value, value + size);                    
                    value += size;
                    
                    break;
                }
            }
            
            return value;
        }
        
        const uint8_t* device_impl::decode_property(device::property& property, const uint8_t* data)
        {
            const protocol::param_info* param_info = reinterpret_cast<const protocol::param_info*>(data);
            
            data += sizeof(protocol::param_info);
            
            property.m_address = param_info->m_value_address;
            property.m_readonly = param_info->m_readonly;
            
            switch(param_info->m_category)
            {
                case(command::VALUE) :
                {
                    property.m_category = device::property::VALUE;
                    data = decode_value(data, static_cast<command::values>(param_info->m_value_type), property.m_value);

                    break;
                }
            
                case(command::RANGE) :
                {
                    data = decode_value(data, static_cast<command::values>(param_info->m_value_type), property.m_value);

                    property.m_category = device::property::RANGE;
                    property.m_attributes.resize(3);
                    
                    data = decode_value(data, static_cast<command::values>(param_info->m_value_type), property.m_attributes[0].first);
                    property.m_attributes[0].second = "min";
                    
                    data = decode_value(data, static_cast<command::values>(param_info->m_value_type), property.m_attributes[1].first);
                    property.m_attributes[1].second = "max";                    
                    
                    data = decode_value(data, static_cast<command::values>(param_info->m_value_type), property.m_attributes[2].first);
                    property.m_attributes[2].second = "step";
                    
                    break;
                }
                
                case(command::SELECTION) :
                {
                    data = decode_value(data, static_cast<command::values>(param_info->m_value_type), property.m_value);                 
                    
                    property.m_category = device::property::SELECTION;
                    
                    uint16_t num = *reinterpret_cast<const uint16_t*>(data);
                    data += sizeof(uint16_t);
                    
                    property.m_attributes.resize(num);
                    
                    for(unsigned int a = 0; a < num; ++a)
                    {
                        data = decode_value(data, static_cast<command::values>(param_info->m_value_type), property.m_attributes[a].first);
                        
                        size_t size = *data;
                        data += sizeof(uint8_t);
                    
                        property.m_attributes[a].second = std::string(data, data + size);
                        data += size;
                    }
                    
                    break;
                }
            }
            
            std::string description((const char*)data, param_info->m_value_description_size);                                    
            data += param_info->m_value_description_size;

            property.m_name = description;
            
            return data;
        }
        
        void device_impl::on_command_received(const boost::system::error_code& error, size_t bytes_received, boost::any& result, lib3dv::error& device_error)
        {
            if(m_log_level > 1) std::cout << std::endl << "[II] lib3dv: " << " " << bytes_received << " data bytes read" << std::endl;

            m_commands_deadline.cancel();
            
            if(bytes_received >= sizeof(protocol::packet_header) && !memcmp(m_commands_payload.c_array(), "#3DV", 4))
            {
                const uint8_t* data = m_commands_payload.c_array();
                const protocol::packet_header* packet_header = reinterpret_cast<const protocol::packet_header*>(data);
                data += sizeof(protocol::packet_header);
                
                if(m_log_level > 1)
                {
                    std::cout << "\t" << "protocol version = " << (int)(packet_header->m_protocol_version) << std::endl;
                    std::cout << "\t" << "guid = " << packet_header->m_guid << std::endl;
                    std::cout << "\t" << "fragment = " << packet_header->m_fragment << " / " << packet_header->m_total_fragments << std::endl;
                    std::cout << "\t" << "payload_type = " << (int)packet_header->m_payload_type << std::endl;
                    std::cout << "\t" << "error = " << (int)packet_header->m_error << std::endl;
                }

                if(!packet_header->m_error)
                {
                    device_error.m_type = static_cast<lib3dv::error::types>(packet_header->m_error);

                    switch(packet_header->m_payload_type)
                    {
                        case protocol::payload::COMMAND :
                        {
                            const protocol::command_header* command_header = reinterpret_cast<const protocol::command_header*>(data);
                            data += sizeof(protocol::command_header);

                            switch(command_header->m_type)
                            {
                                case(command::ENUMERATE_PROPERTIES) :
                                {
                                    std::vector<device::property>& properties = boost::any_cast<boost::reference_wrapper<std::vector<device::property> > >(result);

                                    const protocol::params_info_header* params_info_header = reinterpret_cast<const protocol::params_info_header*>(data);
                                    data += sizeof(protocol::params_info_header);
                                    
                                    for(unsigned int i = 0; i < params_info_header->m_params_num; ++i)
                                    {
                                        device::property property;

                                        data = decode_property(property, data);
                                        
                                        properties.push_back(property);
                                    }
                                    
                                    break;
                                }
                                
                                case(command::GET_PROPERTY) :
                                {
                                    device::property property;
                                    
                                    decode_property(property, data);
                                    
                                    result = property.m_value;
                                    
                                    break;
                                }
                            }
                            
                            break;
                        }
                        
                        default:
                            std::cerr << "[EE] lib3dv: unknown payload type: " << (int)packet_header->m_payload_type << std::endl;
                    }
                } else
                    device_error.m_type = static_cast<lib3dv::error::types>(packet_header->m_error);
            }
        }
        
        std::ostream& operator<< (std::ostream& output, const device_impl& device)
        {
            output << "guid: " << device.m_guid
                << "    address: " << device.m_remote_address
                << "    version: " << (int)device.version().m_protocol
                                    << " / " << (int)device.version().m_framework[device::version_info::MAJOR]
                                    << "." << (int)device.version().m_framework[device::version_info::MINOR]
                                    << "." << (int)device.version().m_framework[device::version_info::STEP]
                                    << " / " << (int)device.version().m_application[device::version_info::MAJOR]
                                    << "." << (int)device.version().m_application[device::version_info::MINOR]
                                    << "." << (int)device.version().m_application[device::version_info::STEP]
                << "    capabilities:" << (device.capabilities()[device::capability::DEPTH_MAPPING] ? " DEPTH_MAPPING" : "")
                                      << (device.capabilities()[device::capability::OBSTACLE_DETECTION] ? " OBSTACLE_DETECTION" : "")
                                      << (device.capabilities()[device::capability::TERRAIN_MAPPING] ? " TERRAIN_MAPPING" : "")
                                      << (device.capabilities()[device::capability::MOTION_ESTIMATION] ? " MOTION_ESTIMATION" : "")
                                      << (device.capabilities()[device::capability::CLASSIFICATION] ? " CLASSIFICATION" : "");

            return output;
        }
    }
}
