#ifndef LIB3DV_DETAIL_DEVICE_IMPL_H
#define LIB3DV_DETAIL_DEVICE_IMPL_H

/* lib3dv/detail/device_impl.h
 *
 * Copyright (C) 2013 VisLab
 *
 * This file is part of lib3dv; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <lib3dv/device.h>
#include <lib3dv/image.h>
#include <lib3dv/terrain.h>
#include <lib3dv/motion.h>
#include <lib3dv/candidate.h>
#include <lib3dv/obstacle.h>
#include <lib3dv/detail/command.h>
#include <lib3dv/detail/raw_data.h>
#include <lib3dv/version.h>

#include <boost/array.hpp>
#include <boost/asio.hpp>
#include <boost/date_time/time_duration.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/signals2.hpp>
#include <boost/thread/thread.hpp>
#include <boost/uuid/uuid.hpp>

#include <map>
#include <stdint.h>
#include <vector>
#include <utility>

namespace lib3dv
{
    namespace detail
    {
        class device_impl
        {
            public:

                static const unsigned int MAX_UDP_PAYLOAD_SIZE = 65507; ///< max admissible UDP payload size [bytes].
                
                static std::vector<device> enumerate(const boost::asio::ip::address_v4& local_address, unsigned short remote_commands_port, uint8_t log_level, error& error, const boost::posix_time::time_duration& timeout);
                
                device_impl(const boost::asio::ip::address_v4& local_address,
                            const boost::asio::ip::address_v4& remote_address, unsigned short remote_commands_port, unsigned short remote_data_port,
                            const boost::uuids::uuid& guid,
                            const std::bitset<device::capability::NUM>& capabilities,
                            const device::version_info& version,
                            uint8_t log_level, error& device_error);
                
                ~device_impl();

                inline bool valid() const { return (m_status == device::status::ONLINE) || (m_status == device::status::TRANSMITTING); }
                
                const boost::uuids::uuid& guid() const { return m_guid; }
    
                const std::bitset<device::capability::NUM>& capabilities() const { return m_capabilities; }
                
                const device::version_info& version() const { return m_version; }
                
                void start_transmission(error& error);
            
                void stop_transmission(error& error);
                
                device::status::types status() const { return m_status; }
                
                std::vector<device::property> enumerate_properties(error& error);
                
                void save_properties(error& error);
                
                void reset_properties(error& error);
                
                void poweroff(error& error);
                
                void reboot(error& error);                

                boost::any get_property_value(uint16_t address, error& error);
            
                void set_property_value(uint16_t address, const boost::any& value, error& error);

                uint64_t connect_image_callback(const boost::function<void (boost::shared_ptr<const lib3dv::image>, uint32_t)>& function);

                uint64_t connect_terrain_callback(const boost::function<void (boost::shared_ptr<const lib3dv::terrain>, uint32_t)>& function);

                uint64_t connect_obstacles_callback(const boost::function<void (boost::shared_ptr<const std::vector<lib3dv::obstacle> >, uint32_t)>& function);
                
                uint64_t connect_motion_callback(const boost::function<void (boost::shared_ptr<const lib3dv::motion>, uint32_t)>& function);

                uint64_t connect_classification_callback(const boost::function<void (boost::shared_ptr<const lib3dv::classification>, uint32_t)>& function);

                LIB3DV_DEPRECATED uint64_t connect_image_callback(const boost::function<void (boost::shared_ptr<lib3dv::image>, uint32_t)>& function);

                LIB3DV_DEPRECATED uint64_t connect_terrain_callback(const boost::function<void (boost::shared_ptr<lib3dv::terrain>, uint32_t)>& function);

                LIB3DV_DEPRECATED uint64_t connect_obstacles_callback(const boost::function<void (boost::shared_ptr<std::vector<lib3dv::obstacle> >, uint32_t)>& function);

                LIB3DV_DEPRECATED uint64_t connect_motion_callback(const boost::function<void (boost::shared_ptr<lib3dv::motion>, uint32_t)>& function);

                LIB3DV_DEPRECATED uint64_t connect_classification_callback(const boost::function<void (boost::shared_ptr<lib3dv::classification>, uint32_t)>& function);

                uint64_t connect_timeout_callback(const boost::function<void (void)>& function);
                

                void disconnect_callback(uint64_t id);

                uint8_t m_log_level;

                boost::posix_time::time_duration m_timeout;
                
                friend std::ostream& operator<< (std::ostream& output, const device_impl& device);
                
            private:

                typedef raw_data<protocol::image_info_header> raw_image_type;
                typedef raw_data<protocol::terrain_info_header> raw_terrain_type;
                typedef raw_data<protocol::obstacles_map_info_header> raw_obstacles_type;
                typedef raw_data<protocol::motion_info_header> raw_motion_type;
                typedef raw_data<protocol::classifier_info_header> raw_candidate_type;
                
                static const size_t DEFAULT_MAX_REASSEMBLY_CONTAINER_SIZE = 100;
                
                void on_command_received(const boost::system::error_code& error, size_t bytes_received, boost::any& result, lib3dv::error& device_error);
                
                void on_command_sent(const boost::system::error_code& error, size_t bytes_received, boost::any& result, lib3dv::error& device_error);
                
                void on_timeout(lib3dv::error& device_error);

                /**
                * @brief Asynchronous UDP data handler.
                *
                * @param [in] error Possible error condition.
                * @param [in] bytes_received The amount of data received during the last read [bytes]
                * @return void
                **/
                void on_data_received(const boost::system::error_code& error, size_t bytes_received);
                                
                template<typename C, typename F, typename H>
                typename C::mapped_type init_reassembly_container(C& container, F& fragments_container, const H& data_info_header, uint32_t key);
                
                template<typename C, typename F>
                typename C::mapped_type fill_reassembly_container(C& container, F& fragments_container, size_t bytes_received, uint32_t key, uint32_t fragment, uint32_t total_fragments);
                
                template<typename C, typename F>
                void cleanup_reassembly_container(C& container, F& fragments_container, size_t max_size = DEFAULT_MAX_REASSEMBLY_CONTAINER_SIZE);
                
                void decode_terrain(boost::shared_ptr<raw_terrain_type> terrain, uint32_t guid);
                
                void decode_obstacles(boost::shared_ptr<raw_obstacles_type> obstacles, uint32_t guid);
                
                void decode_motion(boost::shared_ptr<raw_motion_type> pose, uint32_t guid);

                void decode_classification(boost::shared_ptr<raw_candidate_type> pose, uint32_t guid);

                const uint8_t* decode_property(device::property& property, const uint8_t* data);
                
                void send_command(boost::shared_ptr<lib3dv::detail::command> command, error& error);

                std::map<uint32_t, boost::shared_ptr<raw_image_type> > m_images;                
                std::map<uint32_t, std::vector<boost::shared_ptr<raw_data_fragment> > > m_image_fragments;

                std::map<uint32_t, boost::shared_ptr<raw_terrain_type> > m_terrains;
                std::map<uint32_t, std::vector<boost::shared_ptr<raw_data_fragment> > > m_terrain_fragments;
                
                std::map<uint32_t, boost::shared_ptr<raw_obstacles_type> > m_obstacles;
                std::map<uint32_t, std::vector<boost::shared_ptr<raw_data_fragment> > > m_obstacle_fragments;

                std::map<uint32_t, boost::shared_ptr<raw_motion_type> > m_poses;
                std::map<uint32_t, std::vector<boost::shared_ptr<raw_data_fragment> > > m_pose_fragments;

                std::map<uint32_t, boost::shared_ptr<raw_candidate_type> > m_candidates;
                std::map<uint32_t, std::vector<boost::shared_ptr<raw_data_fragment> > > m_candidates_fragments;


                boost::asio::io_service m_data_socket_io_service;
                boost::thread m_data_socket_io_service_thread;

                boost::asio::io_service m_commands_socket_io_service;         
                
                boost::asio::ip::address_v4 m_local_address;
                boost::asio::ip::address_v4 m_remote_address;
                unsigned short m_remote_commands_port;
                unsigned short m_remote_data_port;
                
                boost::asio::ip::udp::socket m_data_socket;
                boost::asio::ip::udp::socket m_commands_socket;
                
                boost::asio::ip::udp::endpoint m_data_sender_endpoint;
                boost::asio::ip::udp::endpoint m_commands_sender_endpoint;
                
                boost::array<uint8_t, MAX_UDP_PAYLOAD_SIZE> m_data_payload;
                boost::array<uint8_t, MAX_UDP_PAYLOAD_SIZE> m_commands_payload;
                
                boost::asio::deadline_timer m_commands_deadline;
                boost::asio::deadline_timer m_data_deadline;


                boost::signals2::signal<void (const command&)> m_command_signal;

                boost::signals2::signal<void (boost::shared_ptr<const lib3dv::image>, uint32_t)> m_image_signal;
                boost::signals2::signal<void (boost::shared_ptr<const lib3dv::terrain>, uint32_t)> m_terrain_signal;
                boost::signals2::signal<void (boost::shared_ptr<const std::vector<lib3dv::obstacle> >, uint32_t)> m_obstacles_signal;
                boost::signals2::signal<void (boost::shared_ptr<const lib3dv::motion>, uint32_t)> m_motion_signal;
                boost::signals2::signal<void (boost::shared_ptr<const lib3dv::classification>, uint32_t)> m_classification_signal;

                LIB3DV_DEPRECATED boost::signals2::signal<void (boost::shared_ptr<lib3dv::image>, uint32_t)> m_deprecated_image_signal;
                LIB3DV_DEPRECATED boost::signals2::signal<void (boost::shared_ptr<lib3dv::terrain>, uint32_t)> m_deprecated_terrain_signal;
                LIB3DV_DEPRECATED boost::signals2::signal<void (boost::shared_ptr<std::vector<lib3dv::obstacle> >, uint32_t)> m_deprecated_obstacles_signal;
                LIB3DV_DEPRECATED boost::signals2::signal<void (boost::shared_ptr<lib3dv::motion>, uint32_t)> m_deprecated_motion_signal;
                LIB3DV_DEPRECATED boost::signals2::signal<void (boost::shared_ptr<lib3dv::classification>, uint32_t)> m_deprecated_classification_signal;

                boost::signals2::signal<void (void)> m_timeout_signal;

                boost::uuids::uuid m_guid;
                std::bitset<device::capability::NUM> m_capabilities;
                device::version_info m_version;
                
                uint64_t m_max_connection_id;
                std::map<uint64_t, boost::signals2::connection> m_connections;

                device::status::types m_status;
        };
    }
}

#include <lib3dv/detail/device_impl.hpp>

#endif
