#ifndef LIB3DV_DETAIL_PROTOCOL_H
#define LIB3DV_DETAIL_PROTOCOL_H

/* lib3dv/detail/protocol.h
 *
 * Copyright (C) 2013 VisLab
 *
 * This file is part of 3dv-client; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include "declspecs.h"

#include <stdint.h>

namespace protocol
{
    /**
     * @brief Possible payload types.
     **/
    struct payload
    {
        enum types
        {
            IMAGE_INFO_HEADER = 0,      ///< image info header.
            IMAGE_DATA,                 ///< image data.
            TERRAIN_INFO_HEADER,
            TERRAIN_DATA,
            OBSTACLES_MAP_INFO_HEADER,
            OBSTACLES_MAP_DATA,
            COMMAND,
            KEEP_ALIVE,
            MOTION_INFO_HEADER,
            MOTION_DATA,
            CLASSIFICATION_INFO_HEADER,
            CLASSIFICATION_DATA,
            NUM
        };
    };

    struct version
    {
        static const unsigned char MAX_SUPPORTED = 10;
    };

    /**
     * @brief An image info packet.
     **/
    DECLSPEC_PACKED(
    struct image_info_header
    {
        uint32_t m_size;            ///< image data buffer size [bytes].
        uint16_t m_width;           ///< image width [px].
        uint16_t m_height;          ///< image height [px].
        uint8_t m_channels;         ///< image channels.
        uint8_t m_bpp;              ///< image bytes per pixel [Bpp].
        uint8_t m_type;             ///< image type. @see protocol::image_type
        uint8_t m_format;           ///< image format. @see protocol::image_format
        int64_t m_timestamp;        ///< image timestamp. Measures elapsed time from linux epoch (1970-1-1) to the beginning of processing [usec].
    });

    DECLSPEC_PACKED(
    struct terrain_info_header
    {
        uint32_t m_size;            ///< terrain data size [bytes].
        int16_t m_x_min;            ///< minimum x distance [cm].
        uint8_t m_x_cell_num;       ///< x cells number.
        uint8_t m_x_cell_size;      ///< size of a cell, x direction [cm].
        int16_t m_y_min;            ///< minimum y distance [cm].
        uint8_t m_y_cell_num;       ///< y cells number.
        uint8_t m_y_cell_size;      ///< the size of a cell, y direction [cm].
    });
    
    DECLSPEC_PACKED(
    struct obstacles_map_info_header
    {
        uint32_t m_size;            ///< obstacles data size [bytes].
        uint16_t m_obstacles_num;   ///< obstacles number.
    });
    
    DECLSPEC_PACKED(
    struct obstacle_info_header
    {
        uint32_t m_guid;            ///< obstacle guid.
        uint16_t m_stixel_num;      ///< stixels number.
    });
    
    DECLSPEC_PACKED(
    struct classifier_info_header
    {
       uint32_t m_size;
       uint16_t m_candidates_num;
       int64_t m_timestamp;
    });

    DECLSPEC_PACKED(
    struct classifier_data
    {
    	uint32_t m_guid;
        uint16_t m_x0;		///box coordinate in right image.
        uint16_t m_y0;
        uint16_t m_x1;
        uint16_t m_y1;
        uint32_t m_confidence;  ///< confidence of the detection.
        uint8_t m_category;  ///< category.
        //uint32_t m_wp_lb_x;
        //uint32_t m_wp_lb_y;
        //uint32_t m_wp_lb_z;


    });

    DECLSPEC_PACKED(
    struct stixel
    {
        uint16_t m_dx;              ///< stixel size (X axis) [cm].
        uint16_t m_dy;              ///< stixel size (Y axis) [cm].
        int16_t m_x;                ///< stixel X position [cm];
        int16_t m_y;                ///< stixel Y position [cm];
        int16_t m_z;                ///< stixel Z position [cm];
        uint16_t m_height;          ///< stixel height [cm];
    });
    
    DECLSPEC_PACKED(
    struct motion_info_header
    {
        uint32_t m_size;            ///< pose data size [bytes].
        uint16_t m_poses_num;       ///< poses number.
    });


    DECLSPEC_PACKED(
    struct pose_info_header
    {
        uint32_t m_size;            ///< pose data size [bytes].
    	uint8_t m_type;             ///< pose type @see lib3dv::pose
    	uint16_t m_pose_num;        ///< pose element number
        int64_t m_timestamp;        ///< pose timestamp. Measures elapsed time from linux epoch (1970-1-1) to the beginning of processing [usec].
    });

    /**
     * @brief A data packet header.
     **/
    DECLSPEC_PACKED(
    struct data_header
    {
        uint32_t m_offset;      ///< data offset [bytes].
    });
    
    /**
     * @brief A device info packet.
     **/
    DECLSPEC_PACKED(
    struct device_info
    {
        uint8_t m_guid[16];         ///< device guid.
        uint16_t m_capabilities;    ///< device capabilities.
        uint8_t m_application_version[3]; ///< application version (0 -> major, 1 -> minor, 2 -> step).
        uint8_t m_framework_version[3];   ///< application version (0 -> major, 1 -> minor, 2 -> step).
    });    
        
    /**
     * @brief A command packet.
     **/
    DECLSPEC_PACKED(
    struct command_header
    {
        uint32_t m_size;
        uint8_t m_type;
    });
    
    /**
     * @brief A propertys list info header.
     **/
    DECLSPEC_PACKED(
    struct params_info_header
    {
        uint16_t m_params_num;
    });
    
    /**
     * @brief A property info.
     **/
    DECLSPEC_PACKED(
    struct param_info
    {
        uint8_t m_value_type;
        uint16_t m_value_address;
        uint8_t m_category;
        uint8_t m_readonly;
        uint8_t m_value_description_size;
    });
    
    /**
     * @brief A generic packet header.
     **/
    DECLSPEC_PACKED(
    struct packet_header
    {
        uint8_t m_magic[4];
        uint8_t m_protocol_version;
        uint32_t m_guid;
        uint32_t m_fragment;
        uint32_t m_total_fragments;
        uint8_t m_error;
        uint8_t m_payload_type;
    });
}

#endif
