// %pacpus:license{
// This file is part of the PACPUS framework distributed under the
// CECILL-C License, Version 1.0.
// %pacpus:license}
/// @version $Id: ComponentBase.cpp 76 2013-01-10 17:05:10Z kurdejma $

#include <Pacpus/kernel/ComponentBase.h>

#include <Pacpus/kernel/ComponentManager.h>
#include <Pacpus/kernel/Log.h>
#include <Pacpus/kernel/PacpusException.h>

#include <boost/program_options/parsers.hpp>
#include <boost/program_options/variables_map.hpp>
#include <string>
#include <vector>

const bool kPropertyVerboseDefaultValue = false;

namespace po = boost::program_options;
using namespace pacpus;
using namespace std;

vector<string> convertAttributesToArgumentVector(const QDomNamedNodeMap & attributes);
void logVariablesMap(po::variables_map vm);

DECLARE_STATIC_LOGGER("pacpus.core.ComponentBase");

ComponentBase::ComponentBase(const QString & componentName)
    : m_componentName(componentName)
    , m_isActive(false)
    , mIsRecording(true)
    , m_manager(NULL)
    , m_ui(NULL)
    , m_componentState(NOT_MONITORED)
    , mOptionsDescription("Component parameters")
{
    LOG_TRACE("constructor");
    // Get a pointer on the instance of ComponentManager.
    m_manager = ComponentManager::getInstance();
    LOG_INFO("component " << getName() << " was created");

    addParameters()
        ("name", po::value<string>(&mName)->required(), "component name")
        ("type", po::value<string>(&mTypeName)->required(), "component type")
        ("ui", po::value<bool>(&mHasGui)->default_value(false), "whether to show GUI")
        ("verbose", po::value<bool>(&mVerbose)->default_value(false), "set output verbose")
        ("verbosity-level", po::value<int>(&mVerbosityLevel)->default_value(0), "set verbosity level")
        ("recording", po::value<bool>(&mIsRecording)->default_value(false), "whether to record data")
    ;
}

ComponentBase::~ComponentBase()
{  
    LOG_TRACE("destructor");
}

bool ComponentBase::isActive() const
{
    return m_isActive;
}

void ComponentBase::setActive(bool isActive)
{
    m_isActive = isActive;
}

bool ComponentBase::isRecording() const
{
    return mIsRecording;
}

void ComponentBase::setRecording(bool isRecording)
{
    mIsRecording = isRecording;
}

const XmlComponentConfig ComponentBase::xmlParameters() const
{
    return param;
}

int ComponentBase::startComponent()
{
    if (isActive()) {
        LOG_DEBUG("component already started, cannot (re-)start");
        return false;
    }

    setActive(true);
    startActivity();
    
    return true;
}

int ComponentBase::stopComponent()
{
    if (!isActive()) {
        LOG_DEBUG("component already stopped, cannot (re-)stop");
        return false;
    }
    
    setActive(false);
    stopActivity();
    
    return true;
}

void ComponentBase::setState(const COMPONENT_STATE state)
{
    m_componentState = state;
}

// FIXME: this should be const.
ComponentBase::COMPONENT_STATE ComponentBase::getState()
{
    COMPONENT_STATE state = m_componentState;
    if (ComponentBase::NOT_MONITORED != m_componentState) {
        m_componentState = ComponentBase::MONITOR_NOK;
    }
    return state;
}

ComponentBase::COMPONENT_CONFIGURATION ComponentBase::configurationState() const
{
    return m_configurationState;
}

void ComponentBase::setConfigurationState(COMPONENT_CONFIGURATION state)
{
    m_configurationState = state;
}

bool ComponentBase::isConfigured() const
{
    return (m_configurationState == CONFIGURED_OK);
}

QString ComponentBase::getName() const
{
    return m_componentName;
}

ComponentBase::InputsMap & ComponentBase::inputs()
{
    return m_inputs;
}

const ComponentBase::InputsMap & ComponentBase::inputs() const
{
    return m_inputs;
}

ComponentBase::OutputsMap & ComponentBase::outputs()
{
    return m_outputs;
}

const ComponentBase::OutputsMap & ComponentBase::outputs() const
{
    return m_outputs;
}

InputInterfaceBase * ComponentBase::getInput(QString inputName) const
{
    if (inputs().contains(inputName)) {
        return inputs()[inputName];
    }
    LOG_WARN("Component " << getName() << " does not contain input " << inputName);
    return NULL;
}

OutputInterfaceBase * ComponentBase::getOutput(QString outputName) const
{
/*    QList<QString> keys = output.keys();
    for (int i=0; i<keys.size();++i)
        LOG_INFO("Key : " << keys[i])*/;

    if (outputs().contains(outputName)) {
        return outputs()[outputName];
    }
    LOG_WARN("Component " << getName() << " does not containt output " << outputName);
    return NULL;
}

bool ComponentBase::hasGui() const
{
    return mHasGui;
}

bool ComponentBase::isOutputVerbose() const
{
    return mVerbose || (getVerbosityLevel() > 0);
}

int ComponentBase::getVerbosityLevel() const
{
    return mVerbosityLevel;
}

po::options_description_easy_init ComponentBase::addParameters()
{
    return mOptionsDescription.add_options();
}

void ComponentBase::parseParameters(const XmlComponentConfig & cfg)
{
    LOG_INFO("Parsing parameters...");
    LOG_INFO(mOptionsDescription);
    
    vector<string> xargs = convertAttributesToArgumentVector(cfg.getProperties());
    
    po::variables_map vm;
    try {
        po::store(
            po::command_line_parser(xargs)
                .options(mOptionsDescription)
                .allow_unregistered()   // FIXME: temporary only, at term all the components specify all parameters
                .run()
        , vm);
        po::notify(vm);
    } catch (po::error & e) {
        LOG_WARN(e.what());
        throw PacpusException(e.what());
    }

    logVariablesMap(vm);
}

void logVariablesMap(boost::program_options::variables_map vm)
{
    for (po::variables_map::iterator i = vm.begin(); i != vm.end(); ++i) { 
        const po::variable_value& v = i->second;
        if (v.empty()) {
            continue;
        }
        const type_info & type = v.value().type();
        if (type == typeid(string)) {
            const string & val = v.as<string>();
            LOG_INFO(i->first << "=" << val);
        } else if (type == typeid(int)) {
            int val = v.as<int>();
            LOG_INFO(i->first << "=" << val);
        } else if (type == typeid(bool)) {
            int val = v.as<bool>();
            LOG_INFO(i->first << "=" << val);
        }
    }
}

vector<string> convertAttributesToArgumentVector(const QDomNamedNodeMap & attributes)
{
    vector<string> xargs;
    xargs.reserve(attributes.size());

    for (int i = 0; i < attributes.size(); ++i) {
        QDomAttr parameter = attributes.item(i).toAttr();
        if (parameter.isNull()) {
            LOG_WARN("node is not a parameter");
            continue;
        }
        
        QString arg = QString("--") + parameter.name() + "=";
        
        bool shouldAddQuotes = parameter.value().contains(' ');
        if (shouldAddQuotes) {
            arg += '\"';
            arg += parameter.value();
            arg += '\"';
        } else {
            arg += parameter.value();
        }
        
        LOG_DEBUG("parameter: " << arg);
        xargs.push_back(arg.toStdString());
    }

    return xargs;
}
