// %pacpus:license{
// This file is part of the PACPUS framework distributed under the
// CECILL-C License, Version 1.0.
// %pacpus:license}
/// @file
/// @author  Gerald Dherbomez <firstname.surname@utc.fr>
/// @date    January, 2006
/// @version $Id: XmlComponentConfig.h 79 2013-01-14 08:52:06Z kurdejma $
/// @copyright Copyright (c) UTC/CNRS Heudiasyc 2006 - 2013. All rights reserved.
/// @brief Brief description.
///
/// Purpose:    definit la structure XML des paramètres de configuration
///             d'un composant

#ifndef DEF_PACPUS_XMLCOMPONENTCONFIG_H
#define DEF_PACPUS_XMLCOMPONENTCONFIG_H

#include <Pacpus/kernel/PacpusLibConfig.h>

#include <QDomElement>

namespace pacpus {
    
//class QDomNamedNodeMap;
class XmlConfigFile;

/** XmlComponentConfig
 * @brief Defines the XML structure of a component.
 */
class PACPUSLIB_API XmlComponentConfig
{
    friend class ComponentManager;
public:
    /** Ctor of XmlComponentConfig.
     * @param name Name of the ComponentFactory, by convention equal to class name.
     */
    explicit XmlComponentConfig(QString const& name = QString::null);

    /** Dtor of XmlComponentConfig. */
    ~XmlComponentConfig();

    QDomNamedNodeMap getProperties() const;

    /** Add the property @em name to the XML and set its value to @em 0 if it does not exist.
     * @param name Name of the property.
     */
    void addProperty(QString const& name);

    /** Delete a property from the XML.
     * @return False if the property does not exist, false otherwise.
     */
    int delProperty(QString const& name);

    /** Get the value of a property.
     * @param name Name of the property.
     * @param defaultValue Value returned if the property does not exist.
     * @return Value of the property, @em defaultValue otherwise.
     */
    QString getProperty(QString const& name, QString const& defaultValue = QString::null) const;

    /** Get the value of a property as a boolean.
     * @param name Name of the property.
     * @param defaultValue Value returned if the property does not exist.
     * @return Value of the property, @em defaultValue otherwise.
     */
    bool getBoolProperty(QString const& name, bool defaultValue = false) const;

    /** Get the value of a property as an integer.
     * @param name Name of the property.
     * @param defaultValue Value returned if the property does not exist.
     * @return Value of the property, @em defaultValue otherwise.
     */
    int getIntProperty(QString const& name, int defaultValue = 0) const;

    /** Get the value of a property as a double.
     * @param name Name of the property.
     * @param defaultValue Value returned if the property does not exist.
     * @return Value of the property, @em defaultValue otherwise.
     */
    double getDoubleProperty(QString const& name, double defaultValue = 0.0) const;

    /** Set the value of a property.
     * @param name Name of the property.
     * @param value Value to set.
     */
    void setProperty(QString const& name, QString const& value);

    /** Check if a property exists.
     * @param name Name of the property.
     * @return True if the property exists, false otherwise.
     */
    bool hasProperty(QString const& name) const;
    
public:
    // Returns the local QDomElement.
    QDomElement const& getDomElement() const;
    
private:
    // Copy internally the node in the internal QDomElement. 
    void localCopy(const QDomElement & elementToCopy);

    QString const getComponentName() const;
    QString const getComponentType() const;

    QString const getConnectionType() const;
    QString const getConnectionInput() const;
    QString const getConnectionOutput() const;
    int const getConnectionPriority() const;

private:
    QDomElement component_;
    XmlConfigFile * parentDocument_;
};

} // namespace pacpus

#endif // DEF_PACPUS_XMLCOMPONENTCONFIG_H
