################################################################################
#
# CMake script for finding Log4cxx.
# The default CMake search process is used to locate files.
#
# This script creates the following variables:
#  LOG4CXX_FOUND: Boolean that indicates if the package was found
#  LOG4CXX_INCLUDE_DIRS: Paths to the necessary header files
#  LOG4CXX_LIBRARIES: Package libraries
#  LOG4CXX_LIBRARY_DIRS: Path to package libraries
#
################################################################################

INCLUDE (FindPackageHandleStandardArgs)

# See if LOG4CXX_ROOT is not already set in CMake
IF (NOT LOG4CXX_ROOT)
    # See if LOG4CXX_ROOT is set in process environment
    IF ( NOT $ENV{LOG4CXX_ROOT} STREQUAL "" )
        SET (LOG4CXX_ROOT "$ENV{LOG4CXX_ROOT}")
        MESSAGE (STATUS "Detected LOG4CXX_ROOT set to '${LOG4CXX_ROOT}'")
    ENDIF ()
ENDIF ()

SET (LOG4CXX_INC_LIST
  "/usr/include"
  "/usr/local/include"
)

SET (LOG4CXX_LIB_LIST
  "/usr/lib"
  "/usr/local/lib"
)

# If LOG4CXX_ROOT is available, set up our hints
IF (LOG4CXX_ROOT)
    # Includes
    LIST (APPEND LOG4CXX_INC_LIST
      "${LOG4CXX_ROOT}/include"
      "${LOG4CXX_ROOT}"
    )

    # Libraries (win32)
    IF (MSVC)
      # Visual Studio 2008
      IF (MSVC9)
        LIST (APPEND LOG4CXX_LIB_LIST "${LOG4CXX_ROOT}/lib/msvc2008")
      ENDIF ()
      # Visual Studio 2010
      IF (MSVC10)
        LIST (APPEND LOG4CXX_LIB_LIST "${LOG4CXX_ROOT}/lib/msvc2010")
      ENDIF ()
    ENDIF ()

    # Libraries (all)
    LIST (APPEND LOG4CXX_LIB_LIST "${LOG4CXX_ROOT}/lib")
ENDIF ()

# Find headers
FIND_PATH (
  LOG4CXX_INCLUDE_DIR
  NAMES
    log4cxx/log4cxx.h
  HINTS
    ${LOG4CXX_INC_LIST}
)

# Find release library
FIND_LIBRARY (
  LOG4CXX_LIBRARY
  NAMES
    log4cxx
  HINTS
    ${LOG4CXX_LIB_LIST}
)

# Find debug library, if on UNIX this is the same as release
IF (WIN32)
  FIND_LIBRARY (LOG4CXXD_LIBRARY NAMES log4cxx_d HINTS ${LOG4CXX_LIB_LIST})
ELSE ()
  SET (LOG4CXXD_LIBRARY ${LOG4CXX_LIBRARY})
ENDIF ()

# Set LOG4CXX_FOUND honoring the QUIET and REQUIRED arguments
find_package_handle_standard_args(LOG4CXX DEFAULT_MSG LOG4CXX_LIBRARY LOG4CXX_INCLUDE_DIR)

# Output variables
IF (LOG4CXX_FOUND)
  # Include dirs
  SET (LOG4CXX_INCLUDE_DIRS ${LOG4CXX_INCLUDE_DIR})

  # Libraries
  IF (LOG4CXX_LIBRARY)
    SET (LOG4CXX_LIBRARIES optimized ${LOG4CXX_LIBRARY} debug ${LOG4CXXD_LIBRARY})
  ELSE (LOG4CXX_LIBRARY)
    SET (LOG4CXX_LIBRARIES "")
  ENDIF (LOG4CXX_LIBRARY)
  IF (LOG4CXXD_LIBRARY)
    SET (LOG4CXX_LIBRARIES debug ${LOG4CXXD_LIBRARY} ${LOG4CXX_LIBRARIES})
  ENDIF (LOG4CXXD_LIBRARY)

  # Link dirs
  GET_FILENAME_COMPONENT (LOG4CXX_LIBRARY_DIRS ${LOG4CXX_LIBRARY} PATH)
ENDIF ()

# Advanced options for not cluttering the cmake UIs
MARK_AS_ADVANCED (LOG4CXX_INCLUDE_DIR LOG4CXX_LIBRARY)