// %pacpus:license{
// This file is part of the PACPUS framework distributed under the
// CECILL-C License, Version 1.0.
/// @author  Marek Kurdej <firstname.surname@utc.fr>
/// @date    2014-04-07 10:17:22
// %pacpus:license}

#include <Pacpus/PacpusTools/SharedMemory.h>
#include <Pacpus/kernel/Log.h>

#include <boost/assert.hpp>
#include <cstdlib>
#include <iomanip>
#include <sstream>
#if PACPUS_OS_WINDOWS
#   include <windows.h>
#endif

DECLARE_STATIC_LOGGER("pacpus.framwork.SharedMemory")

using namespace pacpus;

//////////////////////////////////////////////////////////////////////////
SharedMemory::SharedMemory(const char* name, int size)
{
    LOG_TRACE("constructor(" << name << ", " << size << ")");
    LOG_INFO("creating shared memory" << "\t"
             << "name=" << name << "\t"
             << "size=" << size
             );

#if PACPUS_OS_WINDOWS
    // semaphore name
    std::stringstream semaphoreNameSs;
    semaphoreNameSs << (const char *) "SemShMem_" << name;
    std::string semaphoreName = semaphoreNameSs.str();
    // event name
    std::stringstream eventNameSs;
    eventNameSs << (const char *) "EvtShMem_" << name;
    std::string eventName = eventNameSs.str();

    LOG_DEBUG("semaphore name = " << semaphoreName);
    LOG_DEBUG("event name = " << eventName);

    semaphore_ = CreateSemaphore(NULL, /* lInitialCount = */ 1, /* lMaximumCount = */ 1, semaphoreName.c_str());
    if (semaphore_ == NULL) {
        LPVOID lpMsgBuf;
        DWORD dw = GetLastError();

        FormatMessage(
            /* flags = */ FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
            /* source = */ NULL,
            /* messageId = */ dw,
            /* languageId = */ MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
            /* out buffer = */ (LPTSTR) &lpMsgBuf,
            /* size = */ 0,
            /* args = */ NULL
            );

        LOG_FATAL("cannot create semaphore protection of shared memory segment '" << name << "'"
            << ". Error: " << GetLastError()
            << ". Message: " << (const char *) lpMsgBuf
            << ". Program will exit"
            ); 
        ::exit(-1); 
    }

    // create the event - autoreset mode
    event_ = CreateEvent(NULL, false, false, eventName.c_str()); 

    // lock the semaphore and try to create the shared memory
    lockMemory(); 
    shMemHandle_ = CreateFileMapping(INVALID_HANDLE_VALUE, NULL,	PAGE_READWRITE,	0, size, name);  
    if (shMemHandle_ == NULL) {
        LOG_FATAL("cannot create shared memory segment '" << name << "'"
            << ". Error: " << GetLastError()
            << ". Program will exit"
            ); 
        ::exit(-1);
    }

    // Map the memory to a local pointer
    shMem_ = MapViewOfFile(shMemHandle_, FILE_MAP_ALL_ACCESS, 0, 0, 0); 
    if (shMem_ == NULL) {
        LOG_FATAL("cannot map the view of file of the shared memory segment '" << name << "'"
            << ". Error: " << GetLastError()
            << ". Program will exit"
            ); 
        ::exit(-1);
    }

    LOG_INFO("created Win32 shared memory '" << name << "'"); 

    unlockMemory();

#elif PACPUS_OS_UNIX

    {
        QString memoryName = QString(name);
        memory_ = new QSharedMemory(memoryName);
    }
    if (!memory_) {
        LOG_FATAL("out of memory: cannot create shared memory object");
        exit(-1);
    }

    bool result = false;
    LOG_DEBUG("trying to create a memory segment");
    if (!memory_->create(size)) {
        if (QSharedMemory::AlreadyExists == memory_->error()) {
            LOG_DEBUG("trying to attach to an existing memory segment");
            if (!memory_->attach()) {
                LOG_ERROR("cannot attach to an existing shared memory segment '" << name << "'"
                          << ". error code: " << memory_->error()
                          << ". error message:" << memory_->errorString().toStdString()
                          );
            } else {
                LOG_INFO("attached shared memory segment '" << name << "'");
                int attachedMemorySize = memory_->size();
                LOG_INFO("attached memory size = " << attachedMemorySize);
                if (attachedMemorySize < size) {
                    LOG_ERROR("attached memory is smaller than requested, unexpected behaviour possible");
                    LOG_INFO("use system tools like 'ipcs' and 'ipcrm' to remove shared memory and re-run");
                }
                result = true;
            }
        } else {
            LOG_ERROR("cannot create shared memory segment '" << name << "'"
                      << ". error code: " << memory_->error()
                      << ". error message:" << memory_->errorString().toStdString()
                      );
        }
    } else {
        LOG_INFO("created shared memory segment '" << name << "'");
        result = true;
    }
    
    /////////////////////////////////////////
    if (result) {
        // create the equivalent Windows event (autoreset mode?)
        QString eventName = "EvtShMem_" + QString(name);
        LOG_DEBUG("creating event '" << eventName << "'...");
        event_ = new QSystemSemaphore(eventName, 0, QSystemSemaphore::Create);
        if (event_) {
            LOG_DEBUG("created event '" << eventName << "'");
        } else {
            LOG_ERROR("cannot create event '" << eventName << "'"
                      );
        }

        // get a pointer to the shared memory segment
        shMem_ = memory_->data();
    } else {
        LOG_FATAL("cannot create shared memory segment '" << name << "'");
        //throw MemoryCreationError();
        exit(memory_->error());
    }

#else

#error UNIMPLEMENTED for this operating system

#endif
}

SharedMemory::~SharedMemory()
{
    LOG_TRACE("destructor");
#if PACPUS_OS_WINDOWS
    // free the semaphore 
    CloseHandle(semaphore_); 
    UnmapViewOfFile(shMem_); 
    CloseHandle(shMemHandle_); 
#elif PACPUS_OS_UNIX
    // detach this process from the shared memory
    memory_->detach();
    // free shared memory
    delete memory_;
    // free event
    delete event_;
#endif
}

bool SharedMemory::wait(unsigned long timeout)
{
#if PACPUS_OS_WINDOWS
    if (timeout == 0) {
        timeout = INFINITE; 
    }

    DWORD status = 0; 
    status = WaitForSingleObject(event_, timeout); 

    if (status == WAIT_OBJECT_0) {
        return true; 
    } else {
        return false; 
    }
#elif PACPUS_OS_UNIX
    (void) timeout; // unused
    bool result = event_->acquire();
    LOG_TRACE("event acquired: note that timeout doesn't work yet on linux platform");
    return result;
#endif
}

void* SharedMemory::read()
{
    void* shMem;
#if PACPUS_OS_WINDOWS
    lockMemory(); 
#endif
    shMem = shMem_;
#if PACPUS_OS_WINDOWS
    unlockMemory(); 
#endif
    return shMem;
}

void SharedMemory::read(void* dst, int size)
{
    lockMemory();
    memcpy(dst, shMem_, size); 
    unlockMemory();
}

void* SharedMemory::getEventIdentifier()
{
#if PACPUS_OS_WINDOWS
    return event_; 
#elif PACPUS_OS_UNIX
    LOG_TRACE("no event identifier");
    return NULL;
#endif
}

void SharedMemory::write(void *data, int size, unsigned long offset)
{
    using std::hex;

    LOG_TRACE("writing " << size << " bytes to shared memory at offset " << offset);
#if PACPUS_OS_WINDOWS
    lockMemory(); 
    //unsigned long * dest = (unsigned long *)shMem_ + offset;
    char * dest = (char *)shMem_ + offset;
    //printf("adresses : shm : %x  dest : %x   offset : %x \n",shMem_, dest, offset); 
    memcpy(dest, data, size);
    unlockMemory();
    SetEvent(event_); 
#elif PACPUS_OS_UNIX
    lockMemory();
    uint32_t * dest = (uint32_t *) shMem_ + offset;

    LOG_TRACE("adresses:"
              << " shm = " << hex << shMem_
              << " dst = " << hex << dest
              << " src = " << hex << data
              );

    memcpy(dest, data, size);
    unlockMemory();
    event_->release();
#endif
}

void SharedMemory::lockMemory()
{
#if PACPUS_OS_WINDOWS
    WaitForSingleObject(semaphore_, INFINITE); 
#elif PACPUS_OS_UNIX
    BOOST_ASSERT(memory_);
    if (!memory_->lock()) {
        LOG_ERROR("cannot lock memory '" << memory_->key().toStdString() << "'");
    }
#endif
}

void SharedMemory::unlockMemory()
{
#if PACPUS_OS_WINDOWS
    ReleaseSemaphore(semaphore_, 1, NULL); 
#elif PACPUS_OS_UNIX
    BOOST_ASSERT(memory_);
    if (!memory_->unlock()) {
        LOG_ERROR("cannot unlock memory '" << memory_->key().toStdString() << "'");
    }
#endif
}
