/**
 *
 * Distributed under the UTC Heudiascy Pacpus License, Version 1.0.
 * Copyright (c) UTC Heudiasyc 2010 - 2013. All rights reserved.
 *
 * See the LICENSE file for more information or a copy at:
 *   http://www.hds.utc.fr/~kurdejma/LICENSE_1_0.txt
 *
 */

#ifndef DEF_PACPUS_DBTPLYENGINE_H
#define DEF_PACPUS_DBTPLYENGINE_H

#include <QSemaphore>

#include <Pacpus/kernel/ComponentBase.h>
#include <Pacpus/kernel/ComponentFactory.h>
#include <Pacpus/DbitePlayer/DbitePlayerConfig.h>
#include <Pacpus/DbitePlayer/DbtPlyEngineStateChart.h>
#include <Pacpus/DbitePlayer/DbtPlyUserInterface.h>

namespace pacpus {

class DbtPlyEngineStateChart;
class DbtPlyEngineState;

enum PlayMode
{
    PlayModeLastData = 1,   // replay_mode="1"
    PlayModeAllData = 2     // replay_mode="2"
};

/// XML config properties:
///     datadir         PATH    root data directory,
///         e.g. datadir="/home/user/dbt_data"
///     replay_mode     STRING  {1}|2
///         e.g. replay_mode="2"
class DBITE_PLAYER_API DbtPlyEngine
        : public QThread
        , public ComponentBase
{
    Q_OBJECT

public:
    /// Constructor
    DbtPlyEngine(QString name);
    /// Destructor
    ~DbtPlyEngine();

    /// @returns the directory where the data are stored
    QString getDataDir();

    /// @returns the current replay time
    road_time_t getTime()
    {
        return tDbt_;
    }

    /// @returns true if the player is playing
    bool isPlaying();
    /// Accessor to playmode
    int replayMode()
    {
        return replayMode_;
    }

    const DbtPlyEngineState * getState();
    void setState(DbtPlyEngineState * newState);

    /// Goes back to the beginning, sets speed to initial value
    void reset();

    void speedUp();
    void speedDown();
    void setLastTNow(road_time_t newTNow)
    {
        this->lastTNow_= newTNow;
    }

protected:
    // The 3 virtual methods relative to the ComponentBase inheritance
    virtual void startActivity();
    virtual void stopActivity();
    virtual COMPONENT_CONFIGURATION configureComponent(XmlComponentConfig config);

    /// Thread main loop
    virtual void run();

private:
    DbtPlyEngineState * mCurrentState;
    float mSpeed;
    bool mIsReverse;

    /// The state chart of the engine (speed, state and direction)
    DbtPlyEngineStateChart * stateChart_;

    /// A semaphore of protection of the critical section
    QSemaphore * tMinMaxSem_;
    /// tmin and tmax regarding to all the DBT files replayed
    road_time_t tDbtMin_, tDbtMax_;

    /// tNow_ = the current time when the trigger alerts the engine
    road_time_t tNow_;
    /// lastTNow_ = previous tNow_
    road_time_t lastTNow_;
    /// tDbt_ = the DBT time at which the data have to be replayed corresponding the state of the engine
    road_time_t tDbt_;
    /// lastTDbt_ = previous tDbt_
    road_time_t lastTDbt_;

    /// the direction of reading: forward=1 or backward=-1 used to compute the tDbt that must be replayed
    short direction_;

    /// the synchronization semaphore with the trigger
    QSemaphore * sync_;

    /// where the data are stored
    QString dataDir_;

    /// mode
    PlayMode replayMode_;

Q_SIGNALS:
    void play(road_time_t timeToPlay, road_time_t actualTime, bool direction);
    void stopfile();
    void displayStateSig(DbtPlyEngineState * state, float speed);
    void timeMinMax(road_time_t tMin, road_time_t tMax);
    void curReplayTime(road_time_t time);

public Q_SLOTS:
    void engReceiver();
    void changeDirection(bool reverse);
    void tMinMax(road_time_t tMin, road_time_t tMax);

    void playEvent();
    void pauseEvent();
    void stopEvent();
    void speedUpEvent();
    void speedDownEvent();
};

} // namespace pacpus

#endif // DEF_PACPUS_DBTPLYENGINE_H
