/**
 *
 * Distributed under the UTC Heudiascy Pacpus License, Version 1.0.
 * Copyright (c) UTC Heudiasyc 2010 - 2013. All rights reserved.
 *
 * See the LICENSE file for more information or a copy at:
 *   http://www.hds.utc.fr/~kurdejma/LICENSE_1_0.txt
 *
 */
 
#ifndef DEF_PACPUS_ASYNC_WORKER_BASE_H
#define DEF_PACPUS_ASYNC_WORKER_BASE_H

// Includes, pacpus.
#include <Pacpus/PacpusTools/utility/NonCopyable.h>

// Includes, qt.
#include <QObject>

namespace pacpus
{
  /**
   * @brief A simple base class for asynchronous workers able to partake in the
   * Qt slot / signal mechanism
   *
   * This class represents an asynchronous event-based worker able to receive and
   * send Qt signals to other workers or threads. The rationale is to be able to
   * define stateful objects that perform calculations asynchronously, triggered
   * by external user-defined messages.
   */
  class AsyncWorkerBase
    : public QObject
    , private NonCopyable
  {
    Q_OBJECT
    public:
      /** Constructor. */
      AsyncWorkerBase();

      /** Destructor. */
      virtual ~AsyncWorkerBase();

      /** Starts the worker by creating a new thread for it, and calling the setup() virtual method. */
      void start();

      /** Terminates the worker as soon as there are no more requests pending and its current processing
       *  is finished.
       *
       *	This method tries to end the worker gracefully. All pending signals sent before the stop() request will
       *	be honored.
       *	It is safe to call this method from any thread.
       *
       *  @param autoDelete if true, deletes the worker as soon as its event loop has finished processing.
       */
      void stop(bool autoDelete);

      /** Returns true if the worker is active. */
      bool isActive() const { return active_; }

    signals:
          void finished();
          void stopped();

    protected:
       virtual void setup();
       virtual void process();
       virtual void cleanup();

       /** Returns true if the worker is stopping. */
       bool isStopping() const { return stopping_; }

    private slots:
      void doSetup();
      void doFinish();

    private:
      /*! \brief Ends the worker, asking the underlying thread to terminate
       *
       * This method signals the end of processing and requests the underlying thread to terminate. Actual termination
       * will occur as soon as all pending signals (including those that may come from other workers during the
       * finish request handling) have been processed.
       */
      void finish();

      // Attributes
      bool active_;
      bool stopping_;
  };
} // namespace pacpus

#endif // DEF_PACPUS_ASYNC_WORKER_BASE_H
