// %pacpus:license{
// This file is part of the PACPUS framework distributed under the
// CECILL-C License, Version 1.0.
// %pacpus:license}
/// @version    $Id: Log.cpp 76 2013-01-10 17:05:10Z kurdejma $

#include <Pacpus/kernel/Log.h>

#if defined(PACPUS_USE_LOG)

#if defined(PACPUS_LOG_COLORED_OUTPUT)
#   include "ColorSeverityFormatter.hpp"
#endif

#include <boost/log/detail/date_time_format_parser.hpp>
#include <boost/log/expressions.hpp>
#include <boost/log/expressions/formatters/if.hpp>
#include <boost/log/sinks/text_file_backend.hpp>
#include <boost/log/sinks/text_ostream_backend.hpp>
#include <boost/log/sources/severity_logger.hpp>
#include <boost/log/support/date_time.hpp>
#include <boost/log/utility/setup/common_attributes.hpp>
#include <boost/log/utility/setup/file.hpp>
#include <boost/log/utility/setup/formatter_parser.hpp>
#if BOOST_VERSION >= 105500 // header exists from 1.55
#   include <boost/utility/empty_deleter.hpp>
#else
#   include <boost/log/utility/empty_deleter.hpp>
#endif
#include <ostream>
#include <QString>

//template <typename CharT>
//PACPUSLIB_API std::basic_ostream<CharT>& operator<< (std::basic_ostream<CharT>& strm, QString const& s)
//{
//    strm << s.toLocal8Bit().data();
//    return strm;
//}

// specialization for char
template <>
PACPUSLIB_API std::basic_ostream<char>& operator<< (std::basic_ostream<char>& strm, QString const& s)
{
    strm << s.toStdString();
    return strm;
}

// specialization for wchar_t
template <>
PACPUSLIB_API std::basic_ostream<wchar_t>& operator<< (std::basic_ostream<wchar_t>& strm, QString const& s)
{
    strm << s.toStdWString();
    return strm;
}

namespace pacpus
{

BOOST_LOG_ATTRIBUTE_KEYWORD(severity, "Severity", ::pacpus::SeverityLevel)

void init_log_factories()
{
    boost::log::register_simple_formatter_factory< QString, char >("QString");
}

static int niftyCounter;

LogConfigurator::LogConfigurator()
{
    if (0 == niftyCounter++) {
        LOG_INFO("LogConfigurator constructor");
        init_log_factories();
    }
}

LogConfigurator::~LogConfigurator()
{
    if (0 == --niftyCounter) {
        // clean up
        LOG_INFO("LogConfigurator destructor");
    }
}

void LogConfigurator::configureLoggerWithFile(const char* logFileName)
{
    using namespace boost;

    namespace logging = boost::log;
    namespace sinks = boost::log::sinks;
    namespace src = boost::log::sources;
    namespace expr = boost::log::expressions;
    namespace attrs = boost::log::attributes;
    namespace keywords = boost::log::keywords;
    
    logging::add_common_attributes();
    logging::core::get()->add_global_attribute(
        "ProcessID",
        attrs::current_process_id());
    logging::core::get()->add_global_attribute(
        "ThreadID",
        attrs::current_thread_id());
    //logging::core::get()->add_global_attribute(
    //    "Scope",
    //    attrs::named_scope());

    logging::core::get()->set_filter
    (
#ifdef NDEBUG
        // release
        severity >= debug
#else
        // debug
        severity >= trace
#endif
    );

    ////////////////////////////////////////////////////////////////////////////////
    // FILE
    logging::add_file_log
    (
        keywords::file_name = logFileName,
        keywords::rotation_size = 10 * 1024 * 1024,
        keywords::time_based_rotation = sinks::file::rotation_at_time_point(0, 0, 0),
        //keywords::format = "%LineID% [%TimeStamp%]: %Message%"
        keywords::format =
        (
            expr::stream
                << std::setfill('0') << std::setw(6) << expr::attr< unsigned int >("LineID")
                //<< " [" << expr::format_date_time< posix_time::ptime >("TimeStamp", date_time::iso_extended_format) << "]"
                << " [" << boost::log::expressions::format_date_time< posix_time::ptime >("TimeStamp", "%Y-%m-%d %T.%f") << "]"
                //<< " [" << std::setw(20) << expr::attr<std::string>("Scope") << ">"
                << " <" << severity << ">"
                << " <" << expr::attr< attrs::current_process_id::value_type >("ProcessID")
                << ":" << expr::attr< attrs::current_thread_id::value_type >("ThreadID") << ">"
                << " " << expr::smessage
        )
    );
    
    ////////////////////////////////////////////////////////////////////////////////
    // CONSOLE
    // Create a backend and attach a couple of streams to it
#if ! (BOOST_VERSION >= 105500)
    using logging::empty_deleter;
#endif
    boost::shared_ptr< sinks::text_ostream_backend > backend =
        make_shared< sinks::text_ostream_backend >();
    backend->add_stream(
        shared_ptr< std::ostream >(&std::clog, empty_deleter())
    );

    // Enable auto-flushing after each log record written
    backend->auto_flush(true);

    // Wrap it into the frontend and register in the core.
    // The backend requires synchronization in the frontend.
    typedef sinks::synchronous_sink< sinks::text_ostream_backend > sink_t;
    shared_ptr< sink_t > sink(new sink_t(backend));
    sink->set_filter
    (
#ifdef NDEBUG
        // release
        severity >= info
#else
        // debug
        severity >= debug
#endif
    );

    sink->set_formatter
    (
        expr::stream
#if defined(PACPUS_LOG_COLORED_OUTPUT)
            << formatSeverityWithColors< SeverityLevel >("Severity")
#endif
            << std::setfill('0') << std::setw(6) << expr::attr< unsigned int >("LineID")
            //<< " [" << expr::format_date_time< posix_time::ptime >("TimeStamp", date_time::iso_extended_format) << "] "
            << " [" << expr::format_date_time< posix_time::ptime >("TimeStamp", "%Y-%m-%d %T.%f") << "] "
            //<< " [" << std::setw(20) << expr::attr<std::string>("Scope") << ">"
            << "<" << severity << ">"
            << " "
            << expr::smessage
#if defined(PACPUS_LOG_COLORED_OUTPUT)
            << formatSeverityWithColors< SeverityLevel >("Severity", /*restoreDefault=*/ true) // Resets the terminal to default.
#endif
    );

    logging::core::get()->add_sink(sink);

    LOG_INFO("logger initialised");
}

} // namespace pacpus

#else // PACPUS_USE_LOG

namespace pacpus
{

    LogConfigurator::LogConfigurator()
    {}

    LogConfigurator::~LogConfigurator()
    {}

    void LogConfigurator::configureLoggerWithFile(const char* /*configFilename*/)
    {}

} // namespace pacpus

#endif // PACPUS_USE_LOG

