/*********************************************************************
//  created:    2014/02/11 - 10:48
//  filename:   SickLMSSensor.h
//
//  author:     Cyril Fougeray
//              Copyright Heudiasyc UMR UTC/CNRS 6599
// 
//  version:    $Id: $
//
//  purpose:    Definition of the SickLMSSensor class
*********************************************************************/

#ifndef SICKLMSSENSOR_H
#define SICKLMSSENSOR_H

#include "Pacpus/kernel/ComponentBase.h"
#include "Pacpus/kernel/DbiteFile.h"

#include "AbstractSickSensor.h"
#include "SickLMSData.h"

#include <fstream>
#include <string>

// Export macro for SickLMS DLL for Windows only
#ifdef WIN32
#   ifdef SICKLMS_EXPORTS
        // make DLL
#       define SICKLMS_API __declspec(dllexport)
#   else
        // use DLL
#       define SICKLMS_API __declspec(dllimport)
#   endif
#else
    // On other platforms, simply ignore this 
#   define SICKLMS_API
#endif

class QEvent;

namespace pacpus {
    
class ShMem;
class SickComponent;


/// The class carrying Sick LMS message.
/** This class is used so that we can store every information sent by a Sick LMS sensor.
 * First, the raw data is stored in \c body. Then, if the message is relevant, \c splitMessage is instanciated in order
 * to parse easily information from the sensor.
 * These data are then decoded and stored in the scanData structure.
 */
class SICKLMS_API MessageLMS
{
public:
    /// Constructor.
    MessageLMS()
    {
        time = 0;
        timerange = 0;
        memset(&data,0,sizeof(data));
    }

    /// Destructor.
    ~MessageLMS(){}

    scanData data; //!< Every needed information about the scan (general info + scan points).

    long msgSize; //!< Size of the message

    std::vector<std::string>* splitMessage; //!< The message is split into an array of string in order to be processed easily.
    char* body;                             //!< Raw data
    road_time_t time;                       //!< Time when the first packet of the message is received.
    road_timerange_t timerange;             //!< Timerange : roughly, time between measurement of a point and the processing step (not implemented).
};


//! The class implenting receiving, decoding and storing process of Sick LMS data.
/**
 * This class can be used as a particular thread to acquire data from Sick LDMRS sensors.
 * The Ethernet interface is used to get data from the sensor. Thus, the goal of this class is to
 * get packets and decode them. Also, it offers the possibility to store all relevant information in
 * two files (.dbt and .utc).
 * It can be managed by SickComponent objects.
 */
class SickLMSSensor : public AbstractSickSensor
{
    Q_OBJECT
public:
    /// Constructor
    SickLMSSensor(QObject *parent);

    /**
     * @brief SickLMSSensor constructor.
     * @param parent Basically, a SickComponent object.
     * @param name Name of the sensor in order to write on .dbt and .utc files and to recognize every sensors used.
     * @param ip The IP address of the remote Sick LMS sensor.
     * @param port The port of the remote Sick LMS sensor.
     * @param recording If \c true, data is recorded into dbt + utc files. Data is not recorded otherwise.
     */
    SickLMSSensor(QObject *parent, QString name, QString ip, int port, int recording);

    /// Destructor
    ~SickLMSSensor();

    void run() {}

    void stopActivity(); /*!< To stop the processing thread. */
    void startActivity(); /*!< To start the processing thread. */

    /**
     * @brief reconstituteMessage reconstitute a complete message from received packets
     * @param packet Raw data coming from the sensor.
     * @param length Length of the raw data received.
     * @param time Time of the last received data.
     *
     * A message starts with a <STX> (0x02 in ASCII) char and ends with <ETX> (0x03 in ASCII).
     */
    void reconstituteMessage(const char * packet, const int length, road_time_t time);

    // decoupe le message en scan
    /**
     * @brief processScanData Parse information and process every needed values.
     * @param msg Carries a message. splitMessage field of MessageLMS must be filled.
     * @return Not used for the moment.
     */
    int processScanData(MessageLMS *msg);

    /**
     * @brief isMessageComplete find the <ETX> character (corresponding to the end of a message).
     * @param packets Raw data.
     * @param size Size of raw data.
     * @return The index of the <ETX> character.
     */
    int isMessageComplete(const char* packets, unsigned int size);

public Q_SLOTS:  
    //Fonction qui permet de trier les données reçues et de les enregistrer dans une structure point, puis dans un fichier binaire .dbt
    void customEvent(QEvent * e);
    void configure();
    
public:
    SickSocket * S_socket; //on déclare pointeur de socket
    // SickComponent * myParent;

private:
    // to recognize between several sensors
    QString name_;

    // The SickLMS IP or hostname
    QString ipaddr_;

    // The SickLMS port
    int port_;

    bool recording_; // Enable storing in DBT

    scanCfg mainCfg;

    void storePendingBytes(road_time_t time);
    void fillDataHeader(MessageLMS & msg);
    void fillScanHeader(MessageLMS & msg);
    void askScanCfg();

    int findSTX(const char* packets, const unsigned int size );
    int splitMessage(MessageLMS* message);
    long xstol(std::string str);

    std::string kSickMemoryName;
    std::string kSickDbtFileName;
    std::string kSickUtcFileName;

    std::list<MessageLMS> msgList;

    MessagePacket pendingBytes;

    // write the data on the disk in the dbt file and the associated binary file
    void writeData(MessageLMS &msg);

    // SickLMS_dbt dbtData_;
    pacpus::DbiteFile dbtFile_;
    std::ofstream dataFile_;

#ifdef SickLMS_SH_MEM
    ShMem * shmem_;
    SickLMS_shMem sickData;
#endif

};

} // namespace pacpus

#endif // SICKLMSSENSOR_H
