// Includes, project.
#include "CVWebcamComponent.hpp"

// Includes, pacpus.
#include <Pacpus/kernel/ComponentFactory.h>
#include <Pacpus/kernel/Log.h>
#include <Pacpus/kernel/DbiteFileTypes.h>
#include <Pacpus/kernel/DbiteException.h>

// Includes, qt.
#include <QDir>
#include <QMetaType>

// Includes, standard.
#include <stdexcept>

using namespace pacpus;

// Declare the logger and register the new component.
DECLARE_STATIC_LOGGER("pacpus.component.cvwebcam");
REGISTER_COMPONENT(CVWebcamComponent, "CVWebcam");

const int DEFAULT_DEVICE     = -1;
const int DEFAULT_WIDTH      = 640;
const int DEFAULT_HEIGHT     = 480;
const int DEFAULT_FRAMERATE  = 15;
const int MAX_LENGTH_NAME    = 512;

CVWebcamComponent::CVWebcamComponent(QString const& name)
  : ComponentBase(name)
  , mDevice(DEFAULT_DEVICE)
  , mWidth(DEFAULT_WIDTH)
  , mHeight(DEFAULT_HEIGHT)
  , mHasUI(false)
  , mFramerate(DEFAULT_FRAMERATE)
  , mIsRecording(false)
  , mImageCounter(0)
  , firstTime (true)
  , mShMem (NULL)
{
  // Register new types for the signal/slot system.
  qRegisterMetaType<cv::Mat>("cv::Mat");
}

CVWebcamComponent::~CVWebcamComponent()
{
}

void
CVWebcamComponent::stopActivity()
{
  mWebcam.close();
  disconnect(&mWebcam, SIGNAL(gotFrame(cv::Mat)), this, SLOT(receiveFrame(cv::Mat)));
  mDBTFile.close();
  if (mShMem) {
     delete mShMem;
     mShMem = NULL;
  }
}

void
CVWebcamComponent::startActivity()
{
  firstTime     = true;
  connect(&mWebcam, SIGNAL(gotFrame(cv::Mat)), this, SLOT(receiveFrame(cv::Mat)));

  try
  {
    QString filename = name() + ".dbt";
    mDBTFile.open(filename.toStdString(), WriteMode, FILE_JPEG, MAX_LENGTH_NAME);

    int periodMSec = static_cast<int>((1.f / mFramerate) * 1E3);
    LOG_INFO("period of the acquisition in msec: " << periodMSec);
    mWebcam.open(mDevice, mWidth, mHeight, periodMSec);
    mWebcam.start();
  }
  catch (std::exception const& e)
  {
    LOG_ERROR(e.what());
  }
}

ComponentBase::COMPONENT_CONFIGURATION
CVWebcamComponent::configureComponent(XmlComponentConfig config)
{
  mDevice       = config.getIntProperty("device", DEFAULT_DEVICE);
  mWidth        = config.getIntProperty("width", DEFAULT_WIDTH);
  mHeight       = config.getIntProperty("height", DEFAULT_HEIGHT);
  mFramerate    = config.getIntProperty("framerate", DEFAULT_FRAMERATE);
  mHasUI        = config.getBoolProperty("ui", false);
  mIsRecording  = config.getBoolProperty("recording", false);

  try
  {
    QDir directory;
    if (!directory.exists(name().toLatin1()) && !directory.mkdir(name().toLatin1()))
      throw std::runtime_error("cannot create the webcam directory");

    mFilenameTemplate = QString("%1/image-%2.jpg").arg(name());
    LOG_INFO("template filename: " << mFilenameTemplate);
  }
  catch (std::exception const& e)
  {
    LOG_ERROR(e.what());
    return ComponentBase::CONFIGURED_FAILED;
  }

  return ComponentBase::CONFIGURED_OK;
}

void
CVWebcamComponent::receiveFrame(cv::Mat frame)
{
  if (mHasUI)
    displayFrame(frame);
  if (mIsRecording)
    saveFrame(frame);
  // create the shared memory
  if (firstTime) {
	  mShMem = new ShMem("IMAGE", frame.step * frame.rows );
	  LOG_INFO(frame.step * frame.cols);
	  firstTime = false;
  }
  // send image in shared memory
  mShMem->write(frame.data, frame.step * frame.rows );
}

void
CVWebcamComponent::displayFrame(cv::Mat const& frame)
{
  cv::imshow("WebcamComponent - Frame", frame);
}

void
CVWebcamComponent::saveFrame(cv::Mat const& frame)
{
  try
  {
    QString filename = mFilenameTemplate.arg(mImageCounter);
    ++mImageCounter;

    if (!cv::imwrite(filename.toStdString(), frame))
      throw std::runtime_error("cannot save the following frame: " + filename.toStdString());

    mDBTFile.writeRecord(road_time(), 0, filename.toStdString().c_str(), MAX_LENGTH_NAME);
  }
  catch (DbiteException & e)
  {
    LOG_ERROR("error writing data: " << e.what());
  }
}