// This file is part of the PACPUS framework distributed under the
// CECILL-C License, Version 1.0.
// 
/// @author  Firstname Surname <firstname.surname@utc.fr>
/// @date    Month, Year
/// @version $Id$
/// @copyright Copyright (c) UTC/CNRS Heudiasyc 2006 - 2013. All rights reserved.
/// @brief Brief description.
///
/// Detailed description.

#ifndef DEF_PACPUS_XMLCOMPONENTCONFIG_H
#define DEF_PACPUS_XMLCOMPONENTCONFIG_H

#include <Pacpus/kernel/pacpus.h>

#include <QDomElement>

namespace pacpus {
    
class XmlConfigFile;

/** XmlComponentConfig
 * @brief Defines the XML structure of a component.
 */
class PACPUSLIB_API XmlComponentConfig
{
    friend class ComponentManager;
public:
    /** Ctor of XmlComponentConfig.
     * @param name Name of the ComponentFactory, by convention equal to class name.
     */
    explicit XmlComponentConfig(const QString& name = QString::null);

    /** Dtor of XmlComponentConfig. */
    ~XmlComponentConfig();

    /** Add the property @em name to the XML and set its value to @em 0 if it does not exist.
     * @param name Name of the property.
     */
    void addProperty(const QString& name);

    /** Delete a property from the XML.
     * @return False if the property does not exist, false otherwise.
     */
    int delProperty(const QString& name);

    /** Get the value of a property.
     * @param name Name of the property.
     * @param defaultValue Value returned if the property does not exist.
     * @return Value of the property, @em defaultValue otherwise.
     */
    QString getProperty(const QString& name, const QString& defaultValue = QString::null) const;

    /** Get the value of a property as a boolean.
     * @param name Name of the property.
     * @param defaultValue Value returned if the property does not exist.
     * @return Value of the property, @em defaultValue otherwise.
     */
    int getBoolProperty(const QString& name, bool defaultValue = false) const;

    /** Get the value of a property as an integer.
     * @param name Name of the property.
     * @param defaultValue Value returned if the property does not exist.
     * @return Value of the property, @em defaultValue otherwise.
     */
    int getIntProperty(const QString& name, int defaultValue = 0) const;

    /** Get the value of a property as a double.
     * @param name Name of the property.
     * @param defaultValue Value returned if the property does not exist.
     * @return Value of the property, @em defaultValue otherwise.
     */
    int getDoubleProperty(const QString& name, double defaultValue = 0.0) const;

    /** Set the value of a property.
     * @param name Name of the property.
     * @param value Value to set.
     */
    void setProperty(const QString& name, const QString& value);

    /** Check if a property exists.
     * @param name Name of the property.
     * @return True if the property exists, false otherwise.
     */
    bool hasProperty(const QString& name) const;

private:
    // Returns the local QDomElement.
    QDomElement qDomElement() const;

    // Copy internally the node in the internal QDomElement. 
    void localCopy(const QDomElement& elementToCopy);

    QString const getComponentName() const;
    QString const getComponentType() const;

private:
    QDomElement component_;
    XmlConfigFile * parentDocument_;
};

} // namespace pacpus

#endif // DEF_PACPUS_XMLCOMPONENTCONFIG_H
