/*********************************************************************
//  created:    2012/03/01 - 14:06
//  filename:   PacpusUDPSocket.cpp
//
//  author:     Pierre Hudelaine
//              Copyright Heudiasyc UMR UTC/CNRS 7253
// 
//  version:    $Id: $
//
//  purpose:    Create network socket if needed in Pacpus
//
*********************************************************************/

#include "PacpusUDPSocket.h" 

#include <qbuffer.h>
#include <qbytearray.h>


using namespace pacpus;


DECLARE_STATIC_LOGGER("pacpus.base.PacpusUDPSocket");


//////////////////////////////////////////////////////////////////////////
// Construct the factory
//////////////////////////////////////////////////////////////////////////
static ComponentFactory <PacpusUDPSocket> sFactory("PacpusUDPSocket");  


//////////////////////////////////////////////////////////////////////////
// Constructeur
//////////////////////////////////////////////////////////////////////////
PacpusUDPSocket::PacpusUDPSocket(QString name)
    : ComponentBase(name) 
{
} 


//////////////////////////////////////////////////////////////////////////
// Destructor
//////////////////////////////////////////////////////////////////////////
PacpusUDPSocket::~PacpusUDPSocket() 
{
}


////////////////////////////////////////////////////////////////////////////////
// AddOutputs
////////////////////////////////////////////////////////////////////////////////
void PacpusUDPSocket::addOutputs()
{
	addOutput<QString, PacpusUDPSocket>("udpSocketOutput");
}


////////////////////////////////////////////////////////////////////////////////
// AddInputs
////////////////////////////////////////////////////////////////////////////////
void PacpusUDPSocket::addInputs()
{
	addInput<QString, PacpusUDPSocket>("udpSocketInput", &PacpusUDPSocket::sendDatagrams);
}


//////////////////////////////////////////////////////////////////////////
// Called by the ComponentManager to pass the XML parameters to the 
// component 
//////////////////////////////////////////////////////////////////////////
ComponentBase::COMPONENT_CONFIGURATION  PacpusUDPSocket::configureComponent(XmlComponentConfig config) 
{ 

	socketType_ = config.getProperty("typeSocket");
	
	if (socketType_ == "client" || socketType_ == "Client")
	{
	    address2send_.setAddress(config.getProperty("address"));
	    port2send_ = config.getProperty("port").toUInt();
	    socketType_ = "client";		
	}
	else if (socketType_ == "server" || socketType_ == "Server")
	{
	    port2bind_ = config.getProperty("port").toUInt();
	    socketType_ = "server";
	}
	else
	{
	    qDebug("typeSocket incorrect, become client");
	    address2send_.setAddress(config.getProperty("address"));
	    port2send_ = config.getProperty("port").toUInt();
	    socketType_ = "client";
	}
	
	return ComponentBase::CONFIGURED_OK; 
} 


//////////////////////////////////////////////////////////////////////////
// Called by the ComponentManager to start the component
//////////////////////////////////////////////////////////////////////////
void PacpusUDPSocket::startActivity() 
{ 
	if (!(udpSocket_ = new QUdpSocket()))
		qFatal("Failed to init UDP socket!");
  
 	if (socketType_ == "server")
	{
	    if (udpSocket_->bind(QHostAddress::Any, port2bind_, QUdpSocket::DontShareAddress))
			qDebug() << "Socket bound for server on port " << port2bind_;
	    else
			qWarning() << "Failed to bind socket for server on port" << port2bind_;
	}
	    
	connect(udpSocket_, SIGNAL(readyRead()), this, SLOT(readPendingDatagrams()));
	
	updSocketOutput_ = getTypedOutput<QString, PacpusUDPSocket>("udpSocketOutput");
}


//////////////////////////////////////////////////////////////////////////
// Send datagram QString
//////////////////////////////////////////////////////////////////////////
void PacpusUDPSocket::sendDatagrams(QString frame) 
{ 
	int sent=0;
	
	if (socketType_ == "client")
	{
	    if ((sent = udpSocket_->writeDatagram(frame.toLocal8Bit(), address2send_, port2send_)) == -1) 
			qDebug() << "Failed to send the frame: " << address2send_ << port2send_ << frame << endl; 
	}
	else if (socketType_ == "server")
	{
	    for (int i = 0; i < listClients.size(); i++)
	    {
			if ((sent = udpSocket_->writeDatagram(frame.toLocal8Bit(), listClients[i]->getAddress(), listClients[i]->getPort())) == -1) 
				qDebug() << "Failed to send the frame: " << listClients[i]->getAddress() << listClients[i]->getPort() << frame << endl; 
	    }
	}
} 


//////////////////////////////////////////////////////////////////////////
// Send datagram QByteArray
//////////////////////////////////////////////////////////////////////////
void PacpusUDPSocket::sendDatagrams(QByteArray frame) 
{ 
	int sent=0;
	
	if (socketType_ == "client")
	{
	    if ((sent = udpSocket_->writeDatagram(frame, address2send_, port2send_)) == -1) 
			qDebug() << "Failed to send the frame: " << address2send_ << port2send_ << frame << endl; 
	}
	else if (socketType_ == "server")
	{
	    for (int i = 0; i < listClients.size(); i++)
	    {
			if ((sent = udpSocket_->writeDatagram(frame, listClients[i]->getAddress(), listClients[i]->getPort())) == -1) 
				qDebug() << "Failed to send the frame: " << listClients[i]->getAddress() << listClients[i]->getPort() << frame << endl; 
	    }
	}
} 


//////////////////////////////////////////////////////////////////////////
// Called when the socket receive a new datagram 
//////////////////////////////////////////////////////////////////////////
void PacpusUDPSocket::readPendingDatagrams()
{
	while (udpSocket_->hasPendingDatagrams())  
	{
		QByteArray datagram; 
		datagram.resize(udpSocket_->pendingDatagramSize()); 
		QHostAddress sender; 
		quint16 senderPort;
    
		if(udpSocket_->readDatagram(datagram.data(), datagram.size(),  &sender, &senderPort) != -1)
		{		    
 		    if (socketType_ == "server")
		    {
				bool flag = false;
		
				for (int i = 0; i < listClients.size(); i++) 
				{ 
					if (listClients[i]->getAddress() == sender && listClients[i]->getPort() == senderPort)
						flag = true;
				}
			
				if (flag == false)
					listClients << new Client(sender, senderPort);
			}
			
			if (updSocketOutput_ && updSocketOutput_->hasConnection())
				updSocketOutput_->send(QString(datagram.data()));
		}
		else
		{
		    printf("Error when reading network frame\n");
		}
	}
}


//////////////////////////////////////////////////////////////////////////
// Called by the ComponentManager to stop the component
//////////////////////////////////////////////////////////////////////////
void PacpusUDPSocket::stopActivity()
{
	udpSocket_->close();
	delete udpSocket_;
}
