// *********************************************************************
//  created:    2008/2/28 - 16:24
//  filename:   structureCanCarmen.h
//
//  author:     Gerald Dherbomez
//              Copyright Heudiasyc UMR UTC/CNRS 6599
//
//  version:    $Id: $
//
//  purpose:    Description of the CAN structures for Carmen
//
// to add:
// 094 - comodos
// 612 - eclairage
// 208 - couple
// 349 - couple boite vitesse
// 552 - heure
//
// to verify:
//  350 and 319 not present in real time acquisition
// *********************************************************************

#ifndef STRUCTURECANCARMEN_H
#define STRUCTURECANCARMEN_H

#include <cstddef>

#include "kernel/cstdint.h"
#include "kernel/road_time.h"

/// Basic structure of a CAN frame
struct CanFrame
{
    static const std::size_t MAX_CAN_MSG_LENGTH = 8;

    uint32_t id;
    uint8_t dlc;
    uint8_t data [MAX_CAN_MSG_LENGTH];
};

/// CAN structure with timestamping
struct TimestampedCanFrame
{
    CanFrame frame;
    road_time_t time;
    road_timerange_t timerange;
};

/// Wheel speed CAN frame
/// corresponding CAN frame = 0x44D
typedef struct
{
    float frontWheelsSpeed;     // mean speed of the front wheels (in km/h)
    float rearLeftWheelSpeed;   // speed of the rear left wheel (in km/h)
    float rearRightWheelSpeed;  // speed of the rear right wheel (in km/h)
    float rpmFrontWheels;       // mean rpm of the front wheels (in tr/min)
} StructWheelSpeed;

typedef struct
{
    road_time_t time;
    road_timerange_t timerange;
    StructWheelSpeed d;
} TimestampedStructWheelSpeed;

/// Suspension CAN frame
/// corresponding CAN frame = 0x350
typedef struct
{
    float frontLeftSuspension;  // suspension clearance front left in mm
    float frontRightSuspension; // suspension clearance front right in mm
    float rearLeftSuspension;   // suspension clearance rear left in mm
    float rearRightSuspension;  // suspension clearance rear right in mm
    float trim;                 // the trim in degree
} StructSuspension;

typedef struct
{
    road_time_t time;
    road_timerange_t timerange;
    StructSuspension d;
} TimestampedStructSuspension;

/// Vehicle Speed CAN frame
/// corresponding CAN frame = 0x38D
typedef struct
{
    /// vehicle speed in km/h (referenced to the wheels)
    float vehicleSpeed;
    /// odometer in m (referenced to the wheels). Internally coded as integer 16 bits. Roll over to 0 when value reachs FFFFh (6553.5m to be verified experimentally).
    float distance;
    /// longitudinal acceleration in m/s2 (referenced to the wheels)
    float longitudinalAcc;
} StructVehicleSpeed;

typedef struct
{
    road_time_t time;
    road_timerange_t timerange;
    StructVehicleSpeed d;
} TimestampedStructVehicleSpeed;

/// corresponding CAN frame = 0x319
typedef struct
{
    int8_t yawAccFailSt;          // ?
    bool latAccCal;             // calibration?
    bool latAccValid;           // flag use/don't use
    bool yawRateCal;            // calibration?
    bool yawRateValid;          // flag use/don't use
    float yawRate;              // yaw rate in deg/s
    float latAcc;               // lateral acceleration in m/s2
    float yawSensorOscFreq;     // in Hz
    uint32_t yawSensorSN;  // ?
} StructRawLateralDynamics;

typedef struct
{
    road_time_t time;
    road_timerange_t timerange;
    StructRawLateralDynamics d;
} TimestampedStructRawLateralDynamics;

/// Steering wheel CAN frame
/// corresponding CAN frame = 0x305
typedef struct
{
    float angle;                // in deg - >0 in trigonometric direction (to the left)
    int32_t rate;                  // in deg/s
    int32_t optRate;               // in deg/s optimized rotation rate of the steering wheel
    int8_t errorCode;             // error code - 4 bits
    bool trim;                  // trim must be at 1, if 0 don't use angle and rate
    bool calibration;           // calibration
    bool sensorState;           // sensor state
    int8_t checksum;              // checksum - 4 bits
} StructSteeringWheel;

typedef struct
{
    road_time_t time;
    road_timerange_t timerange;
    StructSteeringWheel d;
} TimestampedStructSteeringWheel;

/// Lateral dynamics CAN frame
/// corresponding CAN frame = 0x3CD
typedef struct
{
    float yawRate;              // >0 in trigonometric direction in deg/s
    float latAcc;               // >0 to the right in m/s2
    float brakePressure;        // master cylinder pressure in bar
    int8_t braking;               // 2 bits
    bool perfLatAccSens;        // ?
    bool perfYawRateSens;       // ?
    bool latAccUncert;          // ?
    bool yawRateUncert;         // ?
    bool brakeContactHS;        // ?
    bool brakeContact;          // ?
    bool muSplitSituation;      // ?
    bool stopEmissionBicapReq;  // ?
} StructLateralDynamics;

typedef struct
{
    road_time_t time;
    road_timerange_t timerange;
    StructLateralDynamics d;
} TimestampedStructLateralDynamics;

/// corresponding CAN frame = 0x550
typedef struct
{
    float vehicleSpeed;              // > 0 only when driving forward, in km/h
    float yawRate;               // >0 to the right, <0 for left turns, in deg/s
    char highBeamStatus;        // High beams (2 bits) 1=ON, 0=OFF, 2=11b=invalid
    char wipersStatus;        // Wipers(2 bits) 1=ON, 0=OFF, 2=11b=invalid
    char brakeStatus;        // is braking (2 bits) 1=ON, 0=OFF, 2=11b=invalid
    char leftSignalStatus;        // left signal(2 bits) 1=ON, 0=OFF, 2=11b=invalid
    char rightSignalStatus;        // right signal (2 bits) 1=ON, 0=OFF, 2=11b=invalid
} StructCanMobileye;

typedef struct
{
    road_time_t time;
    road_timerange_t timerange;
    StructCanMobileye d;
} TimestampedStructCanMobileye;

/// corresponding CAN frame = 0x094
typedef struct
{
    char lightsButtonStatus;              //
    char indicators;               // 0:none, 1:left, 2:right, 3:invalid
    char wipers;        // 0:pos_0, 1:pos_1, 2:pos_2, 4:pos_3
    bool highBeamsCall;               //
} StructComodos;

typedef struct
{
    road_time_t time;
    road_timerange_t timerange;
    StructComodos d;
} TimestampedStructComodos;

/// corresponding CAN frame = 0x612
typedef struct
{
    bool sidelights;              // feux de position
    bool headlights;               // feux de croisement
    bool highBeams;        // feux de route
    bool rightIndicator;               //
    bool leftIndicator;               //
} StructLighting;

typedef struct
{
    road_time_t time;
    road_timerange_t timerange;
    StructLighting d;
} TimestampedStructLighting;

#endif // STRUCTURECANCARMEN_H
