/// Description of the CAN structures for Robotex car (Renault Fluence ZE)
///
/// @date       created 2012/06/21
/// @author     Gerald Dherbomez
/// @copyright  Heudiasyc UMR UTC/CNRS 7253
/// @version    $Id: $

#ifndef __STRUCTURECANFLUENCEZE_H__
#define __STRUCTURECANFLUENCEZE_H__

#include <cstddef>

#include "kernel/cstdint.h"
#include "kernel/road_time.h"

/// Vehicle speed CAN frame
/// corresponding CAN frame = 0x5D7 - frequency 10 Hz
/// 0.6 km/h minimum threshold
typedef struct
{
    float vehicleSpeed;         ///< low frequency vehicle speed (in km/h)
    int32_t accurateOdometer;   ///< accurate odometer for car equiped with electrical brake (in cm)
} StructFluenceZeLowSpeed;

typedef struct
{
    road_time_t time;
    road_timerange_t timerange;
    StructFluenceZeLowSpeed d;
} TimestampedStructFluenceZeLowSpeed;

/// Front Wheel speed CAN frame
/// corresponding CAN frame = 0x29A - frequency 50 Hz
typedef struct
{
    float frontLeftWheelSpeed;      ///< speed of the front left wheel (in rpm) - min threshold 5 rpm
    float frontRightWheelSpeed;     ///< speed of the front right wheel (in rpm) - idem
    float averageFrontWheelSpeed;   ///< average front wheels speed (in km/h)
} StructFluenceZeFrontWheelSpeed;

typedef struct
{
    road_time_t time;
    road_timerange_t timerange;
    StructFluenceZeFrontWheelSpeed d;
} TimestampedStructFluenceZeFrontWheelSpeed;

/// Rear Wheel speed CAN frame
/// corresponding CAN frame = 0x29C - frequency 50 Hz
typedef struct
{
    float rearLeftWheelSpeed;       ///< speed of the front left wheel (in rpm)
    float rearRightWheelSpeed;      ///< speed of the front right wheel (in rpm)
    float fourWheelReferenceSpeed;  ///< four wheels reference speed (in km/h) available for 4x4 only
} StructFluenceZeRearWheelSpeed;

typedef struct
{
    road_time_t time;
    road_timerange_t timerange;
    StructFluenceZeRearWheelSpeed d;
} TimestampedStructFluenceZeRearWheelSpeed;

/// Steering wheel CAN frame
/// corresponding CAN frame = 0x0C6 - 100 Hz
/// min -3276.7 / max 3276.7 for all values
typedef struct
{
    float angle;                ///< in deg >0 in trigonometric direction (to the left)
    float rotationSpeed;        ///< in deg/s
    float angleOffset;          ///< in deg
} StructFluenceZeSteeringWheel;

typedef struct
{
    road_time_t time;
    road_timerange_t timerange;
    StructFluenceZeSteeringWheel d;
} TimestampedStructFluenceZeSteeringWheel;

/// Vehicle Kinematics CAN frame
/// corresponding CAN frame = 0x12E - frequency 100 Hz
typedef struct
{
    float yawRate;              ///< >0 in trigonometric direction in deg/s (min -204.7deg/s, max 204.7deg/s)
    float transversalAcc;       ///< >0 (??) to the right in g (min -1.8g, max 1.8g)
    float longitudinalAccProc;  ///< >0 (??) to the right in m/s2 (min -10m/s2, max 2.7m/s2)
} StructFluenceZeKinematics;

typedef struct
{
    road_time_t time;
    road_timerange_t timerange;
    StructFluenceZeKinematics d;
} TimestampedStructFluenceZeKinematics;

/// Gears lever position
/// corresponding CAN frame = 0x17E - frequency 100 Hz
typedef struct
{
    enum GearLeverPosition
    {
        Parking = 0
        , Reverse = 1
        , Neutral = 2
        , Drive = 3
    };
    
    uint8_t position;   ///< 0x00 parking ; 0x01 reverse ; 0x02 neutral ; 0x03 drive (in the order of the lever)
} StructFluenceZeGears;

typedef struct
{
    road_time_t time;
    road_timerange_t timerange;
    StructFluenceZeGears d;
} TimestampedStructFluenceZeGears;

#endif 
