#pragma warning( once : 4290 )
#pragma warning( once : 4274 )

#ifndef _SBAS_H
#define _SBAS_H

/*! \class SBAS
 *	\brief Allows to handle SBAS messages 
 *	\author Olivier LE MARCHAND
 *	\version 1.0
 *	\date April 2008
 *	\bug None
 *	\warning None
 *	
 *
 */
namespace SBAS{

//4*4+2 premiers bits = message ID

/*---------------------------------------------------------------------
---------------------------------- MT1 --------------------------------
-----------------------------------------------------------------------*/
/*!\brief Message type 1 - mask*/
/*! Message type 1 is used to transmit PRN mask for all satellites monitored by EGNOS system.
 *  That includes 37 GPS satellites, 24 GLONASS satellites, 58 future GNSS satellites, 19 GEO
 *  satellites and 72 future GNSS/GEO/WAAS/Pseudolites.\n
 *  Time out interval: 600s\n
 *  Maximum update interval: 120s*/
struct structPRNMaskAssignement
{
    //bits 15 à 51 :  mask pour le satellite GPS 1
    uint8_t GPSSatMask[37];

    //bits 52 à 75 :  mask pour le satellite GLONASS
    uint8_t GLONASSSatMask[24];

    //bits 76 à 133 :  mask pour les futures GNSS
    uint8_t futGNSSSatMask[58];

    //bits  134 à 152:  mask pour les satellites SBAS 120 à 138
    uint8_t GEOWAASSatMask[19];

    //bits  153 à 224:  mask pour les satellites SBAS 120 à 138
    uint8_t futAllSatMask[72];

    /*!\brief Issue Of Data PRN Mask*/
    /*! IODP is changed each time the PRNMask is modified\n
 * from bits 225 to 226*/
    uint8_t IODP;
};



/*---------------------------------------------------------------------
---------------------------------- MT2 --------------------------------
-----------------------------------------------------------------------*/
// MT2 : 1 - 13
// MT3 : idem 14 - 26
// MT4 : idem 27 - 39
// MT5 : idem 40 - 51
/*!\brief Message type 2 to 5 - Fast corrections*/
/*! Fast corrections are necessary to correct the fast changing errors such as satellite 
 *  clock error\n
 *  Time out interval: variable\n
 *  Maximum update interval: 60s*/

struct structFastCorrections
{
    /*!\brief Issue Of Data Fast Corrections (IODFj)*/
    /*! Identifies the current fast corrections, where j shows fast corrections message
 *  type (2-5). This parameter is used to associate fast corrections with integrity 
 *  information from message type 6\n
 * from bits 15 to 16*/
    uint8_t IODF;

    /*!\brief Issue Of Data PRN Mask*/
    /*! Refers to the IODP of the message of type one in order to associate PRN mask
 *  with Satellite ID\n
 * from bits 17 to 18*/
    uint8_t IODP;

    // bits 19 à 30 : 12 bits PR correction (m) satellite 1 in PRN Mask valeur à diviser par 32
    // bits 31 à 42 : 12 bits PR correction (m) satellite 2 in PRN Mask
    // bits 43 à 54 : 12 bits PR correction (m) satellite 3 in PRN Mask
    // bits 55 à 66 : 12 bits PR correction (m) satellite 4 in PRN Mask
    // bits 67 à 78 : 12 bits PR correction (m) satellite 5 in PRN Mask
    // bits 79 à 90 : 12 bits PR correction (m) satellite 6 in PRN Mask
    // bits 91 à 102 : 12 bits PR correction (m) satellite 7 in PRN Mask
    // bits 103 à 114 : 12 bits PR correction (m) satellite 8 in PRN Mask
    // bits 115 à 126 : 12 bits PR correction (m) satellite 9 in PRN Mask
    // bits 127 à 138 : 12 bits PR correction (m) satellite 10 in PRN Mask
    // bits 139 à 150 : 12 bits PR correction (m) satellite 11 in PRN Mask
    // bits 151 à 162 : 12 bits PR correction (m) satellite 12 in PRN Mask
    // bits 163 à 174 : 12 bits PR correction (m) satellite 13 in PRN Mask
    double  PRCorrectionPRNMask[13];

    //43*4+2 = 174
    // bits 175 à 178 : 1 octet User differential range estimate (m) satellite 1 in PRN mask valeur correspondante à un tableau
    // bits 179 à 182 : 1 octet User differential range estimate (m) satellite 1 in PRN mask
    // bits 183 à 186 : 1 octet User differential range estimate (m) satellite 1 in PRN mask
    // bits 187 à 190 : 1 octet User differential range estimate (m) satellite 1 in PRN mask
    // bits 191 à 194 : 1 octet User differential range estimate (m) satellite 1 in PRN mask
    // bits 195 à 198 : 1 octet User differential range estimate (m) satellite 1 in PRN mask
    // bits 199 à 202 : 1 octet User differential range estimate (m) satellite 1 in PRN mask
    // bits 203 à 206 : 1 octet User differential range estimate (m) satellite 1 in PRN mask
    // bits 207 à 210 : 1 octet User differential range estimate (m) satellite 1 in PRN mask
    // bits 211 à 214 : 1 octet User differential range estimate (m) satellite 1 in PRN mask
    // bits 215 à 218 : 1 octet User differential range estimate (m) satellite 1 in PRN mask
    // bits 219 à 222 : 1 octet User differential range estimate (m) satellite 1 in PRN mask
    // bits 223 à 226 : 1 octet User differential range estimate (m) satellite 1 in PRN mask
    uint16_t UDREI[13];
};

//structFastCorrections fastCorrectionsExtract(uint8_t decodedDataBlock[32]);



/*---------------------------------------------------------------------
---------------------------------- MT6 --------------------------------
-----------------------------------------------------------------------*/
/*!\brief Message type 6 - */
/*! \n
 *  Time out interval: 18s\n
 *  Maximum update interval: 12s*/
struct structIntegrityInformation
{
    // bits 15 à 16
    /*!\brief Issue of Data Fast correction corresponding to the one used in MT2*/
    uint8_t IODF2;

    // bits 17 à 18 : IODF3
    /*!\brief Issue of Data Fast correction corresponding to the one used in MT3*/
    uint8_t IODF3;

    // bits 19 à 20 : IODF4
    /*!\brief Issue of Data Fast correction corresponding to the one used in MT4*/
    uint8_t IODF4;

    // bits 21 à 22 : IDOF5
    /*!\brief Issue of Data Fast correction corresponding to the one used in MT5*/
    uint8_t IODF5;

    // bits 23 à 27 : UDREI for sat 1 of mask
    //et continue jusqu'au 51
    /*!\brief User Data Differential Estimate Indicator*/
    uint8_t UDREI[51];
};

//structIntegrityInformation IntegrityInformationExtract(uint8_t decodedDataBlock[32]);


/*---------------------------------------------------------------------
---------------------------------- MT7 --------------------------------
-----------------------------------------------------------------------*/
/*!\brief Message type 7 - Fast Correction Degradation Factor*/
/*! \n
 *  Time out interval: 360s\n
 *  Maximum update interval: 120s*/
struct structFastCorrectionDegradationFactor
{
    // bits 15 à 18 : latency in second
    uint8_t systemLatency;

    // bits 19 à 20 : IODP
    uint8_t IODP;

    // bits 23 à 26 : fast correction degradation factor of sat1 of mask
    //et continue jusqu'au 51
    uint8_t ai[51];
};



/*---------------------------------------------------------------------
---------------------------------- MT8 --------------------------------
-----------------------------------------------------------------------*/
/*!\brief Message type 8 - */
/*! \n
 *  Time out interval: \n
 *  Maximum update interval: */
struct structGEONavigationMessage
{
    // bits 15 à 22 : IOD
    uint8_t IOD;

    // bits 23 à 35 : t0
    uint32_t t0;

    // bits 36 à 37 : accuracy URA
    uint16_t accuracyURA;

    // bits 38 à 67 : Xg(ECEF)
    double XgECEF;

    // bits 68 à 97 : Yg(ECEF)
    double YgECEF;

    // bits 98 à 122 : Zg(ECEF)
    double ZgECEF;

    // bits 123 à 139 : Xg rate of change
    double XgRateOfChange;

    // bits 140 à 156 : Yg rateof change
    double YgRateOfChange;

    // bits 157 à 174 : Zg rate of change
    double ZgRateOfChange;

    // bits 175 à 184 : Xg acceleration
    double XgAcceleration;

    // bits 185 à 194 : Yg acceleration
    double YgAcceleration;

    // bits 195 à 204 : Zg acceleration
    double ZgAcceleration;

    // bits 205 à 216 : agf0
    double agf0;

    // bits 217 à 228 : agf1
    double agf1;
};




/*---------------------------------------------------------------------
---------------------------------- MT10 -------------------------------
-----------------------------------------------------------------------*/
/*!\brief Message type 10 - */
/*! \n
 *  Time out interval: 360s\n
 *  Maximum update interval: 120s*/
struct structDegradationParameters
{
    // bits 15 à 24 : Brrc
    double Brrc;

    // bits 25 à 34 : Cltc_lsb
    double Cltc_lsb;

    // bits 35 à 44 : Cltc_v1
    double Cltc_v1;

    // bits 45 à 53 : Lltc_v1
    uint16_t Lltc_v1;

    // bits 54 à 63 : Cltc_v0
    double Cltc_v0;

    // bits 64 à 72 : Lltc_v0
    uint16_t Lltc_v0;

    // bits 73 à 82 : Cgeo_lsb
    double Cgeo_lsb;

    // bits 83 à 92 : Cgeo_v
    double Cgeo_v;

    // bits 93 à 101 : Igeo
    uint16_t Igeo;

    // bits 102 à 107 : Cer
    double Cer;

    // bits 108 à 117 : Ciono_step
    double Ciono_step;

    // bits 118 à 126 : Iiono
    uint16_t Iiono;

    // bits 127 à 136 : Ciono_ramp
    double Ciono_ramp;

    // bits 137 : RSSudre
    bool RSSudre;

    // bits 138 : RSSiono
    bool RSSiono;

    // bits 139 à 145 : Ccovariance
    double Ccovariance;
};


/*---------------------------------------------------------------------
---------------------------------- MT12 -------------------------------
-----------------------------------------------------------------------*/
/*!\brief Message type 12 - */
/*! \n
 *  Time out interval: 1 day, 86400s\n
 *  Maximum update interval: 300s*/
struct structWAASTime
{
    // bits 13 à 36
    double A1wnt;

    // bits 37 à 68
    double A0wnt;

    // bits 69 à 78
    uint32_t t0t;

    // bits 79 à 86
    uint8_t WNt;

    // bits 87 à 94
    int16_t DeltaTLS;

    // bits 95 à 103
    int16_t WNlsf;

    // bits 108 à 110
    uint8_t DN;

    // bits 111 à 118
    int16_t DeltaTLsf;

    // bits 119 à 121
    uint8_t UTSstandardld;

    // bits 122 à 141
    uint32_t GPSTimeOfWeek;

    // bits 142 à 149
    uint16_t GPSWeekNumber;

    // bits  150
    bool GLONASSInd;
};




/*---------------------------------------------------------------------
---------------------------------- MT17 -------------------------------
-----------------------------------------------------------------------*/
/*!\brief Message type 17 - */
/*! \n
 *  Time out interval: none\n
 *  Maximum update interval: 300s*/
struct structGEOSatelliteAlmanacs
{
    // bits
    uint8_t DataID[3];

    // bits
    uint16_t SPRNNumber[3];

    //bits
    int32_t XgECEF[3];

    //bits
    int32_t YgECEF[3];

    //bits
    int32_t ZgECEF[3];

    //bits
    int8_t XgRateOfChange[3];

    //bits
    int8_t YgRateOfChange[3];

    //bits
    int8_t ZgRateOfChange[3];

    //bits
    bool ranging[3];

    //bits
    bool corrections[3];

    //bits
    bool broadcatsIntegrity[3];

    //bits
    bool reserved[3];

    //bits
    uint8_t ID[3];

    //bits
    uint32_t t0;
};




/*---------------------------------------------------------------------
---------------------------------- MT18 --------------------------------
-----------------------------------------------------------------------*/
/*!\brief Message type 18 - Ionospheric Grid Point Mask*/
/*! Message type 18 is used to ....\n
 *  Time out interval: 1200s\n
 *  Maximum update interval: 300s*/


struct structIonosphericGridPointMask
{
    //bits
    /* !\brief Ionosphere Issue of Data*/
    /* ! between 0 and 3*/
    uint8_t IODIonosphere;

    //bits
    /* !\brief Number of broadcasted bands*/
    /* ! Value between 0 and 9. O means that no ionospheric will be provided*/
    uint8_t numberOfBroadcastedBands;

    //bits
    /* !\brief Number of the current band*/
    /* ! Value between 0 and 8. Only value between 3 and 5 are useful for EGNOS*/
    uint8_t bandNumber;

    bool IGPMask[201];
};




/*---------------------------------------------------------------------
---------------------------------- MT24 -------------------------------
-----------------------------------------------------------------------*/

/*!\brief Message type 24 - */
/*! \n
 *  Time out interval: \n
 *  Maximum update interval: */
struct structMixedFastCorrectionLongTermError
{
    //bits
    double PRCorrection[6];

    //bits
    uint8_t UDREI[6];

    //bits
    uint8_t blockID;

    //bits
    bool velocityCode;

    //bits
    uint8_t FCIODF;

    //bits
    uint8_t FCIODP;

    //bits
    uint8_t PRNMask[2];

    //bits
    uint8_t IODE[2];

    //bits
    double deltaXECEF[2];

    //bits
    double deltaYECEF[2];

    //bits
    double deltaZECEF[2];

    //bits
    double deltaXROC;

    //bits
    double deltaYROC;

    //bits
    double deltaZROC;

    //bits
    double deltaAf0[2];

    //bits
    double deltaAf1;

    //bits
    double t0;

    //bits
    uint8_t LTIODP;
};



/*---------------------------------------------------------------------
---------------------------------- MT25 -------------------------------
-----------------------------------------------------------------------*/
/*!\brief Message type 25 - Long Term Corrections*/
/*! \n
 *  Time out interval: 360s\n
 *  Maximum update interval: 120s*/
struct structLongTermSatelliteErrorCorrections
{
    //bits
    bool velocityCode[2];

    //bits
    uint8_t IODP[2];

    //bits
    uint8_t PRNMask[4];

    //bits
    uint8_t IODE[4];

    //bits
    double deltaXECEF[4];

    //bits
    double deltaYECEF[4];

    //bits
    double deltaZECEF[4];

    //bits
    double deltaXROC[4];

    //bits
    double deltaYROC[4];

    //bits
    double deltaZROC[4];

    //bits
    double deltaAf0[4];

    //bits
    double deltaAf1[4];

    //bits
    double t0[2];
};



/*---------------------------------------------------------------------
---------------------------------- MT26 -------------------------------
-----------------------------------------------------------------------*/
/*!\brief Message type 26 - Ionospheric Delay Correction*/
/*! Message type 26 is used to ....\n
 *  Time out interval: 600s\n
 *  Maximum update interval: 300s*/
struct structIonosphericDelayCorrections
{
    /* !\brief Ionosphere Issue of Data*/
    /* ! between 0 and 3*/
    uint8_t IODIonosphere;

    //bits
    /* !\brief Number of the Band*/
    /* ! between 0 and 9*/
    uint8_t bandNumber;

    //bits
    /* !\brief Block Identifier*/
    /* ! between 0 and 13*/
    uint8_t blockID;

    //bits
    /* !\brief Grid Ionospheric Vertical Error Indicator*/
    /* ! between 0 and 15\n
 *   GIVEI = 15 means "not monitored"*/
    uint8_t GIVEI[15];

    //bits
    /* !\brief Vertical delay estimation of the IGP*/
    /* ! between 0 and 63.875
 *   vertical delay = 63.750m means not monitored*/
    double IGPVerticalDelayEstimate[15];
};




/*---------------------------------------------------------------------
---------------------------------- MT27 -------------------------------
-----------------------------------------------------------------------*/
/*!\brief Message type 27- */
/*! \n
 *  Time out interval: none\n
 *  Maximum update interval: 300s*/
struct structWAASServiceMessage
{
    //bits
    uint8_t numberOfRegions;

    //bits
    int8_t coordinate1Latitude[5];

    //bits
    int8_t coordinate1Longitude[5];

    //bits
    int8_t coordinate2Latitude[5];

    //bits
    int8_t coordinate2Longitude[5];

    //bits
    bool regionShape[5];

    //bits
    uint8_t IODService;

    //bits
    uint8_t numberOfServiceMsgs;

    //bits
    bool serviceMessageNumber;

    //bits
    uint8_t priorityCode;

    //bits
    uint8_t UDREIndicatorInside;

    //bits
    uint8_t UDREIndicatorOutside;
};





/*---------------------------------------------------------------------
---------------------------------- MT28 -------------------------------
-----------------------------------------------------------------------*/
/*!\brief Message type 28 - */
/*! \n
 *  Time out interval: \n
 *  Maximum update interval: */
struct structClockEphCovMatrixMsg
{
    //bits
    uint8_t PRNMask[2];

    //bits
    uint8_t ScaleExp[2];

    //bits
    uint16_t E11[2];

    //bits
    uint16_t E22[2];

    //bits
    uint16_t E33[2];

    //bits
    uint16_t E44[2];

    //bits
    int16_t E12[2];

    //bits
    int16_t E13[2];

    //bits
    int16_t E14[2];

    //bits
    int16_t E23[2];

    //bits
    int16_t E24[2];

    //bits
    int16_t E34[2];
};

uint8_t typeIDExtract(uint8_t decodedDataBlock[32]);

uint8_t preambleExtract(uint8_t decodedDataBlock[32]);


inline uint16_t bytesSpread(uint16_t startBit, uint16_t length)
{
    uint16_t nbBytes;
	nbBytes = ((startBit+length-2)/8 - (startBit-1)/8 +1);
	return nbBytes;
}

int8_t extractChar(uint8_t* dataBlock, uint16_t startBit, uint16_t length);

int16_t extractShort(uint8_t* dataBlock, uint16_t startBit, uint16_t length);

int32_t extractLong(uint8_t* dataBlock, uint16_t startBit, uint16_t length);

uint16_t extractUShort(uint8_t* dataBlock, uint16_t startBit, uint16_t length);

uint32_t extractULong(uint8_t* dataBlock, uint16_t startBit, uint16_t length);
}

#endif
