#ifndef _UBX_structures_Ublox_H
#define _UBX_structures_Ublox_H

#define MAX_SBAS_SV 24
#define NB_CHANNEL 16

#include "kernel/road_time.h"

namespace Ublox {

/*! \struct structRAW_SV
  * \brief Structure containing measurement from the GPS receiver for each satellite
  *	\author "Olivier Le Marchand"
  *	\version 0.1
  *	\date February 2008
  *
  * Ublox receiver realize several measures for each satellite, and provide it
  * through the RXM-RAW message. All the measures for one satellite are grouped
  * in this structure\n
 *	Protocols definitions are available in the following document : \n
 *	Title:     ANTARIS Positioning Engine  \n
 *	Subtitle:  NMEA and UBX Protocol Specification  \n
 *	Doc Type:  Specification  \n
 *	Doc ID:    GPS.G3-X-03002-D \n
 *	Firmware:  Version 5.00
 */
struct structRAW_SV
{
    /// Carrier Phase measurement (L1 Cycles)
    double CPMes;

    /// Pseudorange measurement (m)
    double PRMes;

    /// Doppler Measurement (Hz)
    float DOMes;

    /// Satellite number
    uint8_t SV;

    /// Nav measurement Quality indicator
    int8_t MesQI;

    /// Signal Strength (dBHz)
    int8_t CNO;

    /// Loss of Lock indicator
    uint8_t LLI;
};

/*! \struct structSBAS_SV
 *  \brief Structure containing the SBAS correction data for one satellite
 *	\author "Olivier Le Marchand"
 *	\version 0.1
 *	\date February 2008
 *
 *	Protocols definitions are available in the following document : \n
 *	Title:     ANTARIS Positioning Engine  \n
 *	Subtitle:  NMEA and UBX Protocol Specification  \n
 *	Doc Type:  Specification  \n
 *	Doc ID:    GPS.G3-X-03002-D \n
 *	Firmware:  Version 5.00
 */
struct structSBAS_SV
{
    /// SV Id
    uint8_t SVID;

    /// Flags for this SV
    uint8_t flags;

    /// Monitoring status
    uint8_t UDRE;

    /// System (WAAS/EGNOS/...) same as sys
    uint8_t SYSn;

    /// Services available sames sevices
    uint8_t serviceN;

    /// reserved
    uint8_t res0;

    /// Pseudo Range correction in [cm]
    int16_t PRC;

    /// reserved
    int16_t res1;

    /// Ionosphere correction in [cm]
    int16_t IC;
};

/*! \struct structRXM_RAW
  * \brief Structure containing all the measurement from the GPS receiver at one epoch
  *	\author "Olivier Le Marchand"
  *	\version 0.1
  *	\date February 2008
  *
  * Ublox receiver realize several measures for each satellite, and provide it
  * through the RXM-RAW message. All the measures for all satellites are grouped
  * in this structure in order to pass all throw a RTMaps wire\n
 *	Protocols definitions are available in the following document : \n
 *	Title:     ANTARIS Positioning Engine  \n
 *	Subtitle:  NMEA and UBX Protocol Specification  \n
 *	Doc Type:  Specification  \n
 *	Doc ID:    GPS.G3-X-03002-D
  */
struct structRXM_RAW
{
    /// Measurement in ms GPS time  of Week
    int32_t ITOW;

    /// Measurement GPS Week numbers
    int16_t Week;

    /// number of Satellite in the frame
    uint8_t NSV;

    /*! \brief measurement for each satellite */
    /*! the size of the array is 16, because hte ublox receiver we have can't track
 more than 16 satellites*/
    structRAW_SV tabRAW_SV[NB_CHANNEL];
};

/*! \struct Timestamped_RXM_RAW
 * \brief Timestamped structure containing all the measurement at one epoch, for RT processing and fusion
 *	\author "Vincent Drevelle"
 *	\date June 2011
  */
struct Timestamped_RXM_RAW
{
    road_time_t time;
    road_timerange_t timerange;
    structRXM_RAW raw;
};

/*! \struct structRXM_SFRB
 * \brief Structure containing of one single frame buffer
 *	\author "Olivier Le Marchand"
 *	\version 0.1
 *	\date February 2008
 *
 * \n
 *	Protocols definitions are available in the following document : \n
 *	Title:     ANTARIS Positioning Engine  \n
 *	Subtitle:  NMEA and UBX Protocol Specification  \n
 *	Doc Type:  Specification  \n
 *	Doc ID:    GPS.G3-X-03002-D
 */
struct structRXM_SFRB
{
    /// Channel number
    uint8_t CHN;

    /// ID of the satellite transmitting the subframe
    uint8_t SVID;

    //array of ten words of data
    uint32_t DWRD[10];
};

/*! \struct Timestamped_RXM_SFRB
 * \brief Timestamped structure containing one single frame buffer, for RT processing and fusion
 *	\author "Vincent Drevelle"
 *	\date June 2011
 */
struct Timestamped_RXM_SFRB
{
    road_time_t time;
    road_timerange_t timerange;
    structRXM_SFRB sfrb;
};

/*! \struct structNAV_CLOCK
 * \brief Structure containing the estimation of the clock bias and the clock drift
 *	\author "Olivier Le Marchand"
 *	\version 0.1
 *	\date February 2008
 *
 * \n
 *	Protocols definitions are available in the following document : \n
 *	Title:     ANTARIS Positioning Engine  \n
 *	Subtitle:  NMEA and UBX Protocol Specification  \n
 *	Doc Type:  Specification  \n
 *	Doc ID:    GPS.G3-X-03002-D
  */
struct structNAV_CLOCK
{
    /// GPS Millisecond Time of week (ms)
    uint32_t ITOW;

    /// clock bias in nanoseconds
    int32_t CLKB;

    /// clock drift in nanoseconds per second
    int32_t CLKD;

    /// Time Accuracy Estimate
    uint32_t TAcc;

    /// Frequency Accuracy Estimate
    uint32_t FAcc;

};

/*! \struct structNAV_SBAS
 * \brief Structure containing the status of the SBAS system and corrections
 *	\author "Olivier Le Marchand"
 *	\version 0.1
 *	\date February 2008
 *
 * \n
 *	Protocols definitions are available in the following document : \n
 *	Title:     ANTARIS Positioning Engine  \n
 *	Subtitle:  NMEA and UBX Protocol Specification  \n
 *	Doc Type:  Specification  \n
 *	Doc ID:    GPS.G3-X-03002-D
  */
struct structNAV_SBAS
{
    /// GPS Millisecond time of week
    uint32_t ITOW;

    /// PRN Number of the GEO where correction and integrity data is used from
    uint8_t GEO;

    /// SBAS Mode | 0 Disabled | 1 Enabled Integrity | 3 Enabled Testmode
    uint8_t mode;

    /// SBAS System (WAAS/EGNOS/...)  | -1 Unknown | 0 WAAS | 1 EGNOS | 2 MSAS | 16 GPS
    int8_t sys;

    /// SBAS Services available  | bit0 Ranging | bit1 Corrections | bit2 Integrity | bit3 Testmode |
    uint8_t service;

    /// Number of SV data following
    uint8_t CNT;

    /// Reserved
    uint8_t res[3];

    structSBAS_SV tabSBAS_SV[MAX_SBAS_SV];

};

/*! \struct structNAV_SOL
 * \brief Structure containing the position and velocity estimation in the ECEF reference frame
 *	\author "Olivier Le Marchand"
 *	\version 0.1
 *	\date February 2008
 *
 * \n
 *	Protocols definitions are available in the following document : \n
 *	Title:     ANTARIS Positioning Engine  \n
 *	Subtitle:  NMEA and UBX Protocol Specification  \n
 *	Doc Type:  Specification  \n
 *	Doc ID:    GPS.G3-X-03002-D
  */
struct structNAV_SOL
{
    /// GPS Millisecond Time of Week
    uint32_t ITOW;

    /// Nanoseconds remainder of rounded ms above, range -500000 .. 500000
    int32_t Frac;

    /// GPS week (GPS time)
    int16_t week;

    /// GPSfix Type, range 0..4  | 0x00 = No Fix | 0x01 = Dead Reckoning only | 0x02 = 2D-Fix | 0x03 = 3D-Fix | 0x04 = GPS + dead reckoning combined | 0x05 = Time only fix | 0x06..0xff: reserved |
    uint8_t GPSfi;

    /// 0x01=GPSfixOK (i.e. within DOP & ACC Masks) | 0x02=DiffSoln (is DGPS used) | 0x04=WKNSET (is Week Number valid) | 0x08=TOWSET (is Time of Week valid) | 0x?0=reserved
    uint8_t Flags;

    /// ECEF X coordinate (cm)
    int32_t ECEF_X;

    /// ECEF Y coordinate (cm)
    int32_t ECEF_Y;

    /// ECEF Z coordinate (cm)
    int32_t ECEF_Z;

    /// 3D Position Accuracy Estimate (cm)
    uint32_t Pacc;

    /// ECEF X velocity (cm/s)
    int32_t ECEFVX;

    /// ECEF Y velocity (cm/s)
    int32_t ECEFVY;

    /// ECEF Z velocity (cm/s)
    int32_t ECEFVZ;

    /// Speed Accuracy Estimate (cm/s)
    uint32_t SAcc;

    /// Position DOP
    uint8_t PDOP;

    /// reserved
    uint8_t res1;

    /// Number of SVs used in Nav Solution
    uint8_t numSV;

    /// reserved
    uint32_t res2;
};

/*! \struct structSVINFO_SV
 * \brief Structure containing the main information on one satellite
 *	\author "Olivier Le Marchand"
 *	\version 0.1
 *	\date February 2008
 *
 * \n
 *	Protocols definitions are available in the following document : \n
 *	Title:     ANTARIS Positioning Engine  \n
 *	Subtitle:  NMEA and UBX Protocol Specification  \n
 *	Doc Type:  Specification  \n
 *	Doc ID:    GPS.G3-X-03002-D
  */
struct structSVINFO_SV
{

    /// channel number, range 0..NCH-1
    uint8_t chn;

    /// Satellite ID
    uint8_t SVID;

    /*! \brief Bitmask, made up of the following bit values
  *  0x01 = SV is used for navigation
  *  0x02 = Differential correction data is available for this SV
  *  0x04 = Orbit information is available for this SV (Ephemeris or Almanach)
  *  0x08 = Orbit information is Ephemeris
  *  0x10 = SV is unhealthy / shall not be used*/
    uint8_t Flags;

    /*! \brief Signal Quality indicator (range 0..7). The following list shows the meaning of the different QI values:
  * 0: This channel is idle
  * 1, 2: Channel is searching
  * 3: Signal detected but unusable
  * 4: Code Lock on Signal
  * 5, 6: Code and Carrier locked
  * 7: Code and Carrier locked, receiving 50bps data  */
    int8_t QI;


    /// Carrier to Noise Ratio (Signal Strength) (dbHz)
    uint8_t CNO;

    /// Elevation in integer degrees
    int8_t Elev;

    /// Azimuth in integer degrees
    int16_t Azim;

    /// Pseudo range residual in centimetres
    int32_t PRRes;

};

/*! \struct structNAV_SVINFO
 * \brief Structure containing the main information on all the visible satellites
 *	\author "Olivier Le Marchand"
 *	\version 0.1
 *	\date February 2008
 *
 * \n
 *	Protocols definitions are available in the following document : \n
 *	Title:     ANTARIS Positioning Engine  \n
 *	Subtitle:  NMEA and UBX Protocol Specification  \n
 *	Doc Type:  Specification  \n
 *	Doc ID:    GPS.G3-X-03002-D
  */
struct structNAV_SVINFO
{
    /// GPS Millisecond time of week
    uint32_t ITOW;

    /// Number of channels range 0..16
    uint8_t NCH;

    /// Reserved
    uint8_t RES1;

    /// Reserved
    uint16_t RES2;

    /// information for each Satellite
    structSVINFO_SV infoSV[NB_CHANNEL];

};

/*! \struct structNAV_VELNED
 * \brief Structure containing the velocity and heading estimations in a local tangent reference frame
 *	\author "Olivier Le Marchand"
 *	\version 0.1
 *	\date February 2008
 *
 * \n
 *	Protocols definitions are available in the following document : \n
 *	Title:     ANTARIS Positioning Engine  \n
 *	Subtitle:  NMEA and UBX Protocol Specification  \n
 *	Doc Type:  Specification  \n
 *	Doc ID:    GPS.G3-X-03002-D
  */
struct structNAV_VELNED
{
    /// GPS Millisecond Time of Week
    uint32_t ITOW;

    /// NED north velocity (cm/s)
    int32_t VEL_N;

    /// NED east velocity (cm/s)
    int32_t VEL_E;

    /// NED down velocity (cm/s)
    int32_t VEL_D;

    /// Speed (3-D) (cm/s)
    uint32_t Speed;

    /// Ground Speed (2-D)(cm/s)
    uint32_t GSpeed;

    /// Heading 2-D (deg)
    int32_t Heading;

    /// Speed Accuracy Estimate (cm/s)
    uint32_t SAcc;

    /// Course / Heading Accuracy Estimate (deg)
    uint32_t CAcc;
};

/*! \struct structNAV_POSLLH
 * \brief Structure containing the position estimation in the WGS84 coordinate system
 *	\author "Olivier Le Marchand"
 *	\version 0.1
 *	\date February 2008
 *
 * \n
 *	Protocols definitions are available in the following document : \n
 *	Title:     ANTARIS Positioning Engine  \n
 *	Subtitle:  NMEA and UBX Protocol Specification  \n
 *	Doc Type:  Specification  \n
 *	Doc ID:    GPS.G3-X-03002-D
  */
struct structNAV_POSLLH
{
    /// GPS Millisecond Time of Week
    uint32_t ITOW;

    /// Longitude (deg)
    int32_t LON;

    /// Latitude (deg)
    int32_t LAT;

    /// Height above Ellipsoid (mm)
    int32_t HEIGHT;

    /// Height above mean sea level (mm)
    int32_t HMSL;

    /// Horizontal Accuracy Estimate (mm)
    uint32_t Hacc;

    /// Vertical Accuracy Estimate (mm)
    uint32_t Vacc;
};

/*! \struct structNAV_POSUTM
 * \brief Structure containing the position estimation in the UTM coordinate system
 *	\author "Olivier Le Marchand"
 *	\version 0.1
 *	\date February 2008
 *
 * \n
 *	Protocols definitions are available in the following document : \n
 *	Title:     ANTARIS Positioning Engine  \n
 *	Subtitle:  NMEA and UBX Protocol Specification  \n
 *	Doc Type:  Specification  \n
 *	Doc ID:    GPS.G3-X-03002-D
  */
struct structNAV_POSUTM
{
    /// GPS Millisecond Time of Week
    uint32_t ITOW;

    /// UTM Easting (cm)
    int32_t EAST;

    /// UTM Northing (cm)
    int32_t NORTH;

    /// Altitude (cm)
    int32_t ALT;

    /// UTM Zone Number
    int8_t ZONE;

    /// Hemisphere Indicator (0=North, 1=South)
    int8_t HEM;
};

} // namespace Ublox

#endif //  _UBX_structures_Ublox_H
