#ifndef STRUCTURE_TELEMETRE_H
#define STRUCTURE_TELEMETRE_H

#include "kernel/cstdint.h"
#include "kernel/road_time.h"

#if QT_VERSION
#include <QVariant>
#endif

////////////////////////////////////////////////////////////////////////////////
// some constants defined for the Alasca component
static const uint32_t MAX_SCAN_POINT = 8648;
static const uint32_t MAX_SCAN_POINT_PER_LAYER = 2162;
static const int32_t UTC_MAGIC_WORD = 0x55544300;

// Data recorded in the DBITE file
// Alasca XT laserscanner
typedef struct
{
    int8_t scannertype;
    uint32_t timeStart;	  // time start of the scan
    float startAngle;			      // the start angle of the measurement (in [rad*e4])
    float endAngle;			        // the stop angle of the measurement (in [rad*e4])
    uint32_t nbPoint;		  // number of points
    int32_t dataPos;               // the position of the data in the binary file associated to the dbt file
} AlascaXT;

// An Alasca point
// see Manual_Alasca.pdf page 27
// Orientation of the frame regarding the car:
// X ahead, Y on the left and Z upside
struct ScanPoint
{
    uint8_t scannerId;    // ID of the scanner that has detected the point - Alasca has type 2
    uint8_t layerNumber;  // channel (0 = bottom channel ...)
    uint8_t echoNumber;	  // subChannel (0 = A, 1 = B ...)
    uint8_t pointStatus;  // ground, rain, dirt ...
    int16_t x;                    // X coordinate in centimeters
    int16_t y;                    // Y coordinate in centimeters
    int16_t z;                    // Z coordinate in centimeters
    uint16_t width;       // the echo width
};

// Data obtained after decoding
// The complete structure is written in shared memory
// Only the point[MAX_SCAN_POINT] array is recorded in the binary file
// associated to the DBT file (see struct AlascaXT)
struct ScanAlascaData
{
    uint8_t scannertype;        // Alasca has type 2
    uint32_t timeStart;	        // time start of the scan
    float startAngle;			            // the start angle of the measurement (in [rad*e4])
    float endAngle;			              // the stop angle of the measurement (in [rad*e4])
    uint32_t nbPoint;		        // number of points
    road_time_t time;                 // DBT timestamp
    road_timerange_t timerange;       // DBT timerange
    ScanPoint point[MAX_SCAN_POINT];  // the data, see struct ScanPoint
};

struct SortedScanAlascaData
{
    double xYellow[MAX_SCAN_POINT_PER_LAYER];
    double yYellow[MAX_SCAN_POINT_PER_LAYER];
    double xGreen[MAX_SCAN_POINT_PER_LAYER];
    double yGreen[MAX_SCAN_POINT_PER_LAYER];
    double xBlue[MAX_SCAN_POINT_PER_LAYER];
    double yBlue[MAX_SCAN_POINT_PER_LAYER];
    double xRed[MAX_SCAN_POINT_PER_LAYER];
    double yRed[MAX_SCAN_POINT_PER_LAYER];

    int32_t totalSize;
    int32_t yellowSize;
    int32_t greenSize;
    int32_t blueSize;
    int32_t redSize;
};

/// Sick LMS 221 laserscanner
typedef struct
{
    static const std::size_t DATA_COUNT = 101;
    float data[DATA_COUNT];
} TELEMETRE_100;

/// Sick LMS 291 laserscanner
typedef struct
{
    static const std::size_t DATA_COUNT = 181;
    float data[DATA_COUNT];
} TELEMETRE_180;

// cf. page 23
// faire une structure avec le nb de secteurs

// pour chaque secteur :
// donnees statiques : timestamp debut et fin - angles debut et fin - pas de l'angle nb de pts dans le secteur
// tableaux => donnees de base : distance direction et echo amplitude
typedef struct
{
    // TODO
} SickLdoemData;

#if QT_VERSION

Q_DECLARE_METATYPE(ScanPoint)
inline QDataStream & operator << (QDataStream & out, const ScanPoint & Valeur)
{
    out << Valeur.scannerId << Valeur.layerNumber << Valeur.echoNumber << Valeur.pointStatus <<
           Valeur.x << Valeur.y << Valeur.z << Valeur.width;
}

inline QDataStream & operator >> (QDataStream & in, ScanPoint & Valeur)
{
    in >> Valeur.scannerId >> Valeur.layerNumber >> Valeur.echoNumber >> Valeur.pointStatus >>
           Valeur.x >> Valeur.y >> Valeur.z >> Valeur.width;
}
Q_DECLARE_METATYPE(ScanAlascaData)
inline QDataStream & operator << (QDataStream & out, const ScanAlascaData & Valeur)
{
    out << ((quint64)Valeur.time) << Valeur.timerange << Valeur.timeStart << Valeur.scannertype <<
           Valeur.startAngle << Valeur.endAngle << Valeur .nbPoint;
    for (int i =0; i< Valeur.nbPoint;i++)
        out << Valeur.point[i];
    return out;
}

inline QDataStream & operator >> (QDataStream & in, ScanAlascaData & Valeur)
{
    in >> (quint64&)Valeur.time >> Valeur.timerange >> Valeur.timeStart >> Valeur.scannertype >>
           Valeur.startAngle >> Valeur.endAngle >> Valeur .nbPoint;
    for (int i =0; i< Valeur.nbPoint;i++)
        in >> Valeur.point[i];

    return in;
}

#endif

#endif // STRUCTURE_TELEMETRE_H
