/// Purpose Interface with dc1394
///
/// @date created:    2011/02/28 - 15:17
/// @author:     Sergio Rodriguez
/// @version:    $Id: Camera1394Unix.h srodrigu $
///
/// Below are the format, mode and framerate supported by the driver:
///
/// -----------------------------------------
/// Format  Mode  Image Size    Color Format
/// -----------------------------------------
/// 0       0     160x120       YUV(4:4:4)
///         1     320x240       YUV(4:2:2)
///         2     640x480       YUV(4:1:1)
///         3     640x480       YUV(4:2:2)
///         4     640x480       RGB
///         5     640x480       Mono (8-bit)
///         6     640x480       Mono (16-bit)
/// -----------------------------------------
/// 1       0     800x600       YUV(4:2:2)
///         1     800x600       RGB
///         2     800x600       Mono (8-bit)
///         3     1024x768      YUV(4:2:2)
///         4     1024x768      RGB
///         5     1024x768      Mono (8-bit)
///         6     800x600       Mono (16-bit)
///         7     1024x768      Mono (16-bit)
/// -----------------------------------------
/// 2       0     1280x960      YUV(4:2:2)
///         1     1280x960      RGB
///         2     1280x960      Mono (8-bit)
///         3     1600x1200     YUV(4:2:2)
///         4     1600x1200     RGB
///         5     1600x1200     Mono (8-bit)
///         6     1280x960      Mono (16-bit)
///         7     1600x1200     Mono (16-bit)
/// -----------------------------------------
/// 7       0-7   Partial Scan  Selectable
/// -----------------------------------------
///
/// ------------------
/// Code    Rate (fps)
/// ------------------
/// 0        1.875
/// 1       3.75
/// 2       7.5
/// 3       15
/// 4       30
/// 5       60
/// ------------------

#ifndef CAMERA1394UNIX_H
#define CAMERA1394UNIX_H

#include <iostream>
#include <qthread.h>
#include "kernel/ComponentBase.h"
#include "kernel/DbiteFile.h"
#include "ImageViewer.h"

#include <dc1394/conversions.h>
#include <dc1394/control.h>
#include <dc1394/utils.h>

#define MAX_CHAR_PICTURE_NAME		  64
#define MAX_CHAR_DIRECTORY_NAME		64
#define TRUE true
#define FALSE false

class QImage;
class QMutex;

namespace pacpus {

class Camera1394
        : public QThread
        , public ComponentBase
{
  Q_OBJECT

public:
	Camera1394(QString name);
	~Camera1394();
  void run(); 

  bool fillRGB32Image(unsigned char * buffer);
  void YUV444toRGB32(unsigned char* buffer);
  void YUV422toRGB32(unsigned char *buffer); 
  void YUV411toRGB32(unsigned char* buffer);
  void YtoRGB32(unsigned char * buffer);
  void Y16toRGB32(unsigned char * buffer);    // not tested, no camera with this mode available
  void RGB16toRGB32(unsigned char * buffer);  // not tested, no camera with this mode available
  void RGB24toRGB32(unsigned char * buffer);

  inline void startAcquiring()  { acquiring_ = TRUE;  }
  inline void stopAcquiring()   { acquiring_ = FALSE; }
  inline void startRecording()  { recording_ = TRUE;  }
  inline void stopRecording()    { recording_ = FALSE; }

  virtual void stopActivity();  /*!< to stop the processing thread */
  virtual void startActivity(); /*!< to start the processing thread */
  virtual ComponentBase::COMPONENT_CONFIGURATION configureComponent(XmlComponentConfig config); 
  
  // return the period in µs corresponding to the framerate parameter 
  double getPeriod(); 
  bool recordImage(QString format);

signals:
  void newImage(QImage *);
  void corruptedImage();

protected:
  //C1394Camera * theCamera;
  dc1394camera_t*       theCamera;
  dc1394video_mode_t 	  theCamera_video_mode;
  dc1394video_frame_t*  theFrame;
  dc1394framerate_t     theFramerate;
  int                   width_;
  int                   height_;

  QImage * qimage_; 

private:
  // Initialize the camera
  bool initCamera(); 

  // return true if the period T (µs) corresponds to the framerate (at +/-20%)
  bool isPeriodGood(int T); 

  void addInputOutput();

  // properties
  int cameraNode_; 
  int videoFormat_; 
  int videoMode_; 
  int frameRate_;
  bool reverseImage_;
  bool acquiring_;
  bool displaying_;
  bool recording_;
  int zoom_;

  int imageCounter_;
  QString recordFormat_;

  QMutex * imageMutex_;

  road_time_t time_; 
  road_time_t prevTime_; 
  road_timerange_t tr_;

  DbiteFile file_;
  QString dbtFileName_;

  ImageViewer * viewer_;
};

} // namespace pacpus

#endif // CAMERA1394UNIX_H
