/*************************************************************************** 
*
* Copyright 2000 by David Demirdjian.   All rights reserved. 
*  
* Developed  by David Demirdjian
*  
* Permission to use, copy, or modify this software and  its documentation 
* for  educational  and  research purposes only and without fee  is hereby 
* granted, provided  that this copyright notice and the original authors's 
* names appear  on all copies and supporting documentation.  If individual 
* files are  separated from  this  distribution directory  structure, this 
* copyright notice must be included.  For any other uses of this software, 
* in original or  modified form, including but not limited to distribution 
* in whole or in  part, specific  prior permission  must be  obtained from 
* MIT.  These programs shall not  be  used, rewritten, or  adapted as  the 
* basis  of  a  commercial  software  or  hardware product  without  first 
* obtaining appropriate  licenses from David Demirdjian.  The author makes 
* no representations about the suitability of this software for any purpose.  
* It is provided "as is" without express or implied warranty. 
*  
**************************************************************************/
#ifndef _STEREOMATCHING_H
#define _STEREOMATCHING_H

#include "stereobuffer.h"
#include "datastructures.h"
#include "reconst3d.h"


// ********************************************************************
// ********************************************************************
// StereoMatching: class encapsulating all the stereo estimation algos
// ********************************************************************
// ********************************************************************
class StereoMatching {
public:
	// constructor/destructor
	StereoMatching();
	~StereoMatching();

	// processing data at one scale only or using multiple scale
	enum tScaleProcessing { MONOSCALE, MULTISCALE };

	// array of buffers corresponding to different scale levels
	StereoBuffer* sbuffer;

	// context definition: create all buffers for stereo processing
	void initializeContext();
	// destroy all buffers
	void destroyContext();

	// compute input images as given scale 'sc'
	void setInputImages(int sc, const uchar* iml8_bw,  const uchar* imr8_bw);
	void setInputImages(int sc, const uchar* iml8_bw,  const uchar* imr8_bw,
						const uchar* imt8_bw);

	// set/get image size
	void setInputImageSize(int w, int h);
	int getWidth() const;
	int getHeight() const; 

	// set threshold to set a pixel disparity as defined/undefined
	void setAcceptDisparityThreshold(const float newVal);
	float getAcceptDisparityThreshold() const;

	// set the number of scales
	void setNumScales (const int numscales);
	int getNumScales() const;
	// set the (log) scale (scale at which the stereo image is returned)
	// eg. if scale = n, the stereo image will be 2^n twice as small as the input images
	//     if scale = 0, the stereo and input images will have same size
	void setScale (const int scale);
	int getScale() const;

	// for monoscale (standard) or multiscale
	void setScaleProcessing(tScaleProcessing sprocessing);
	tScaleProcessing getScaleProcessing(void) const;

	// set horopter
	void setHoropter (const int horopt);
	int getHoropter() const;
	// set the range of disp to look for ... from 8 to 64 (step. 8)
	void setNumDepth (const int nDepth);
	int getNumDepth() const;

	// set correlation window size from 5 to 17
	void setCorrelationWindowSize(const int hmaskX, const int hmaskY);
	int getCorrelationWindowSizeX() const;
	int getCorrelationWindowSizeY() const;
		
	// *****************************************************
	// doStereo: performs stereo using 2 or 3 images
	// input:
	// - leftImage, rightImage, (topImage): arrays of unsigned char containing the
	//   intensity (grayscale) values of an image
	// output:
	// - simage: a pointer on a StereoImage structure containing the stereo information
	void doStereo(StereoImage* simage, 
				  const unsigned char* leftImage, const unsigned char* rightImage, 
				  const unsigned char* topImage = NULL);

	// -------- monoscale algorithm
	// performs stereo at a given scale
	void doStereo_fixedScale(StereoImage* simage, int scale, const InputImages* inputImage);

	// -------- multiscale algorithm
	// assumes simageHigh and simageLow are 2 stereo images estimated at some consecutive scales
	void doStereo_multiscale(StereoImage* simage, 
							 const StereoImage* simageHigh, const StereoImage* simageLow, 
							 const unsigned char acceptNew);

	// -------- perform sub-pixel estimation -------- 
	void getList_subPixel(StereoImage* sdata, int scale, const StereoBuffer* sb = NULL);

	// check if (x,y,d) is a valid hypothesis
	bool checkValidity(short x, short y, unsigned char d, const char tol) const;

private:
	int width, height; // input image size
	int horopter;	   // translation to apply to left image
	int numScales;		   
	int scale;			// eg. scale = 2 -> disparity image twice as smaller as input images
	int nbDepth, maxNbDepth; // range to search disp.

	// type of processing (monoscale/multiscale)
	tScaleProcessing scaleProcessing;

	// parameters for the (SAD) stereo algorithm 
	int maskSizeX, maskSizeY;	// sizes of SAD filter
	float peakValidationThreshold;		

	// for sub-pixel estimation
	bool subPixelPerformed;
	float *scorePrev, *scoreInit, *scoreNext;
	float *scorePrev_origin, *scoreInit_origin, *scoreNext_origin;

	// internal input images
	InputImages* inputImages;

	void estimateStereo(const unsigned char* iml8_bw, const unsigned char* imr8_bw,  const unsigned char* imtop8_bw,
						int width, int height,
						int maskSizeX, int maskSizeY,  char minDiff, int nbDepth, int nbPartitions,
						StereoBuffer* sb, StereoImage* sdata);

	void estimateStereo_sse2(const unsigned char* iml8_bw, const unsigned char* imr8_bw,  const unsigned char* imtop8_bw,
						int width, int height,
						int maskSizeX, int maskSizeY,  char minDiff, int nbDepth, int nbPartitions,
						StereoBuffer* sb, StereoImage* sdata);

	void estimateStereo_Horiz(const unsigned char* iml8_bw, const unsigned char* imr8_bw, 
							int width, int height,
							int maskSizeX, int maskSizeY,  char minDiff, int nbDepth, int nbPartitions,
							StereoBuffer* sb, StereoImage* sdata);
	void estimateStereo_Horiz_mmx(const unsigned char* iml8_bw, const unsigned char* imr8_bw, 
							int width, int height,
							int maskSizeX, int maskSizeY,  char minDiff, int nbDepth, int nbPartitions,
							StereoBuffer* sb, StereoImage* sdata);

	/// estimate subpixel disparities 'd' on a list of 'nbPoints' image points (x,y)
	void getList_subPixel(const short* x, const short* y,  float* subpixel_depth_list, int nbPoints, 
						  const StereoBuffer* sb, unsigned char* imDepth, int scale);
	// -------- region growing algo -------- 
	// iterative region growing algorithm: 
	// - mode (algo)	 {0,1}
	// - acceptNew		threshold to accept a previously undefined pixel as defined
	// - nbIterations	nb iteration of the algo 
	// - imDisp_start	contains the initial disp image to start with
	//					if not given, the internal disp. image is used as starting image
	void doStereo_grow(StereoImage* simage, int mode, const unsigned char acceptNew, int nbIteration);
	// return the diff. between best corr. score and the associated with (x,y,d) ... return -1 if not in the image
	short checkValidity_error(short x, short y, unsigned char d, const char tol) const;
};

#endif


