// %pacpus:license{
// This file is part of the PACPUS framework distributed under the
// CECILL-C License, Version 1.0.
// %pacpus:license}
/// @version    $Id: XmlConfigFile.cpp 76 2013-01-10 17:05:10Z kurdejma $

#include <Pacpus/kernel/XmlConfigFile.h>
#include <Pacpus/kernel/Log.h>

#include <QFile>
#include <QTextStream>

using namespace pacpus;
using namespace std;

DECLARE_STATIC_LOGGER("pacpus.core.XmlConfigFile");

XmlConfigFile * XmlConfigFile::_xmlConfigFile = NULL;

static const string kPropertyPluginList = "list";

static const string kXmlConfigFilename = "pacpus_config.xml";

#define rootSection "pacpus"
#define componentSection "components"
#define componentNode "component"
#define connectionSection "connections"
#define connectionNode "connection"
#define parameterSection "parameters"
#define pluginNode "plugin"
#define nameAttribute "name"
#define libAttribute "lib"

XmlConfigFile::XmlConfigFile()
    : _file(NULL)
    , _numberOfComponents(0)
{
    LOG_TRACE("constructor");

    // create the root of the XML tree
    _document.appendChild(_document.createElement(rootSection));
    // create the sections
    _document.documentElement().appendChild(_document.createElement(parameterSection));
    _document.documentElement().appendChild(_document.createElement(componentSection));
    _document.documentElement().appendChild(_document.createElement(connectionSection));
    _file = new QFile(kXmlConfigFilename.c_str());
    if (NULL != _file) {
        LOG_INFO("XML document " << kXmlConfigFilename.c_str() << " was created");
    } else {
        LOG_WARN("cannot create XML document " << kXmlConfigFilename.c_str());
    }
}

XmlConfigFile::~XmlConfigFile()
{
    LOG_TRACE("destructor");
}

XmlConfigFile * XmlConfigFile::create()
{
    if (NULL ==_xmlConfigFile) {
        _xmlConfigFile = new XmlConfigFile();
    }
    return _xmlConfigFile;
}

void XmlConfigFile::destroy()
{
    delete _xmlConfigFile;
    _xmlConfigFile = NULL;
}

void XmlConfigFile::addComponent(QDomElement component)
{ 
    _mutex.lock();

    if (_document.documentElement().namedItem(componentSection).namedItem(component.tagName()).isNull()) {
        LOG_WARN("component " << component.tagName() << " exists already in the document");
    } else {
        QDomNode node = _document.documentElement().namedItem(componentSection).appendChild(component);
        ++_numberOfComponents;
        LOG_INFO("component " << node.nodeName() << " has been added to the section "
                 << _document.documentElement().namedItem(componentSection).nodeName());
    }
    _mutex.unlock();
}

void XmlConfigFile::delComponent(QDomElement component)
{
    _mutex.lock();

    QDomNode node = _document.documentElement().namedItem(componentSection).removeChild(component);
    if (node.isNull()) {
        LOG_WARN("component " << component.tagName() << " doesn't exist in the document.");
    } else {
        LOG_INFO("component " << node.nodeName() << " has been removed from the section "
                 << _document.documentElement().namedItem(componentSection).nodeName());
        --_numberOfComponents;
    }

    _mutex.unlock();
}

QDomElement XmlConfigFile::createComponent(QString name)
{
    LOG_DEBUG("creating component " << name);

    QMutexLocker mutexLocker(&_mutex);
    Q_UNUSED(mutexLocker);
    return _document.createElement(name);
}

void XmlConfigFile::saveFile(QString fileName)
{
    QMutexLocker mutexLocker(&_mutex);
    Q_UNUSED(mutexLocker);

    _file->setFileName(fileName);
    {
        _file->open(QIODevice::WriteOnly);
        {
            QTextStream ts(_file);
            ts << "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>\n" + _document.toString();
        }
        _file->close();
    }
    LOG_DEBUG("file \"" << _file->fileName() << "\" has been saved");
}

int XmlConfigFile::loadFile(QString fileName)
{
    QMutexLocker mutexLocker(&_mutex);
    Q_UNUSED(mutexLocker);

    int line = 0, col = 0;
    QString errorMsg;

    if (_numberOfComponents != 0) {
        LOG_WARN("XML document contained " << _numberOfComponents << " components that will be lost!");
    }

    _file->setFileName(fileName);
    if (!_file->open( QIODevice::ReadOnly )) {
        LOG_ERROR("cannot open file " << fileName);
        return 0;
    }
    if (!_document.setContent( _file, true, &errorMsg, &line, &col )) {
        LOG_ERROR("cannot get data into the file " << fileName);
        LOG_ERROR(errorMsg << " at position " << line << ":" << col << " (line:col)");
        _file->close();
        return 0;
    }
    _file->close();

    // get the number of components in the loaded tree
    _numberOfComponents = _document.documentElement().namedItem(componentSection).childNodes().count();

    LOG_INFO("XML file \"" << fileName << "\" has been loaded. Number of components = " << _numberOfComponents);
    LOG_DEBUG("XML file content:\n"
              << "BEGIN============================================================================\n"
              << _document.toString()
              << "END==============================================================================\n"
              );

    return _numberOfComponents;
}

QDomElement XmlConfigFile::getComponent(QString componentName)
{
    LOG_DEBUG("getting component " << componentName);

    QDomNodeList nodeList = _document.documentElement().namedItem(componentSection).toElement().elementsByTagName(componentNode);
    for (int i = 0; i < _numberOfComponents; i++) {

        if(nodeList.at(i).toElement().attribute(nameAttribute) == componentName)
            return nodeList.at(i).toElement();
    }

    LOG_WARN("cannot get component " << componentName << ": document does not contain the component");

    return QDomElement();
} 

QStringList XmlConfigFile::getAllComponentsNames()
{
    QStringList componentNameList;   
    QDomNodeList nodeList = _document.documentElement().namedItem(componentSection).toElement().elementsByTagName(componentNode);
     for (int i = 0; i < _numberOfComponents; i++) {
         componentNameList.append(nodeList.at(i).toElement().attribute(nameAttribute));
    }
    return componentNameList;
}

QDomNodeList XmlConfigFile::getAllComponents()
{
    return _document.documentElement().namedItem(componentSection).toElement().elementsByTagName(componentNode);
}

QStringList XmlConfigFile::getAllPlugins()
{
    QDomNodeList nodeList = _document.documentElement().namedItem(parameterSection).toElement().elementsByTagName(pluginNode);
    QStringList stringList;
    if(nodeList.size() == 0) {
        LOG_WARN("no plugins were specified");
    } else
        for(int i = 0; i< nodeList.size(); ++i)
            stringList.append(nodeList.at(i).toElement().attribute(libAttribute));

    return stringList;
}

QDomNodeList XmlConfigFile::getAllConnections()
{
   return _document.documentElement().namedItem(connectionSection).childNodes();
}
