/*********************************************************************
// created:    2011/05/18
// filename:   QSerialPort.cpp
//
// author:     Gerald Dherbomez
// 
// version:    $Id: $
//
// purpose:    This class defines a unix (32bits) driver for serial port
 *********************************************************************/

#include "Pacpus/PacpusTools/QSerialPort.h"

#include <QDebug>

#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <termios.h>
#include <unistd.h>
#include <sys/ioctl.h>


QSerialPort::~QSerialPort() {
    delete serialPort;
}

QSerialPort::QSerialPort(QString name) {
    serialPort = new QextSerialPort(name, QextSerialPort::EventDriven);
}


//////////////////////////////////////////////////////////////////////////
// return the number of frames that have not been yet decoded 
//////////////////////////////////////////////////////////////////////////

int QSerialPort::numberOfFrames() {
    if (dataFrameList.isEmpty())
        return 0;
    return dataFrameList.count();
}

//////////////////////////////////////////////////////////////////////////
// return a pointer to the first frame in the list 
//////////////////////////////////////////////////////////////////////////

FRAME * QSerialPort::firstFrame() {
    return dataFrameList.first();
}


//////////////////////////////////////////////////////////////////////////
// remove the first frame of the list 
// use this function as soon as you copy the frame 
//////////////////////////////////////////////////////////////////////////

int QSerialPort::removeFirstFrame() {
    frameLock_.lock();

    FRAME * tmp = dataFrameList.first();
    if (tmp == NULL) {
        qWarning("Not possible to delete the first item of the dataframe list. It is empty.");
        return 0;
    }
    dataFrameList.removeFirst();
    delete[] tmp->data;
    delete tmp;
    tmp = NULL;

    frameLock_.unlock();

    return 1;
}

/*!< open the port 'name'
return true if success 
 */
bool QSerialPort::openPort(const char * name) {
/*    handlePort = open(name, O_RDWR | O_NOCTTY | O_NDELAY);
    if (handlePort == -1) {
        qWarning("openPort : Unable to open serial port %s",name);
        return false;
    } else {
        fcntl(handlePort, F_SETFL, 0);
        printf("Port %s has been sucessfully opened and % d is the file description\n", name, handlePort);
        return true;
    }*/

    serialPort->setPortName(name);

    if (serialPort->open(QIODevice::ReadWrite)) {
        connect(serialPort, SIGNAL(readyRead()), this, SLOT(RecieveData()));

        if (!(serialPort->lineStatus() & LS_DSR))
            qDebug() << "warning: device is not turned on";

        qDebug() << "listening for data on" << serialPort->portName();
        return true;
    } else {
        qDebug() << "device failed to open:" << serialPort->errorString();
        return false;
    }
}


/*!< close the port
return true if success 
 */
int QSerialPort::closePort() {
    serialPort->close();
}

/*!< configure the port 
return true if success 
 */
int QSerialPort::configurePort(long baudRate, int byteSize, char parity, int stopBits) {

    switch (baudRate) {
        case 4800:
            serialPort->setBaudRate(BAUD4800);
            break;      
        case 9600:
            serialPort->setBaudRate(BAUD9600);
            break;
        case 19200:
            serialPort->setBaudRate(BAUD19200);
            break;
        case 38400:
            serialPort->setBaudRate(BAUD38400);
            break;
        case 115200:
            serialPort->setBaudRate(BAUD115200);
            break;
        default:
            serialPort->setBaudRate(BAUD9600);
            break;
    }

    switch (byteSize) {
        case 5:
            serialPort->setDataBits(DATA_5);
            break;
        case 6:
            serialPort->setDataBits(DATA_6);
            break;
        case 7:
            serialPort->setDataBits(DATA_7);
            break;
        case 8:
            serialPort->setDataBits(DATA_8);
            break;
        default:
            serialPort->setDataBits(DATA_8);
            break;
    }

    switch (parity) {
        case 0:
            serialPort->setParity(PAR_NONE);
            break;
        case 1:
            serialPort->setParity(PAR_EVEN);
            break;
        case 2:
            serialPort->setParity(PAR_ODD);
            break;
        case 3:
            serialPort->setParity(PAR_SPACE);
            break;
        default:
            serialPort->setParity(PAR_NONE);
            break;
    }

    switch (stopBits) {
        case 1:
            serialPort->setStopBits(STOP_1);
            break;
        case 2:
            serialPort->setStopBits(STOP_2);
            break;
        default:
            serialPort->setStopBits(STOP_1);
            break;
    }

    serialPort->setFlowControl(FLOW_OFF);
}



//////////////////////////////////////////////////////////////////////////
// Read 'maxLength' bytes on the port and copy them in buffer 
// return the number of bytes read
//////////////////////////////////////////////////////////////////////////
int QSerialPort::readBuffer(char *buffer, int maxLength)
{
  int countread = read( handlePort,buffer,maxLength);
  //printf("read %d of %d, data:%s\n", countread, maxLength, buffer);
  return countread;
}


void QSerialPort::run()
{
  while (THREAD_ALIVE)
  {
    numberBytesToRead = nbBytesToRead();
    if (numberBytesToRead > 0)
    {
      t_ = road_time();                     // datation
      receivedBuffer_ = new char[numberBytesToRead];
      memset(receivedBuffer_,0,numberBytesToRead);
      numberBytesRead = readBuffer(receivedBuffer_,numberBytesToRead);
      processIncomingData(); 
    }
    else
    {
      receivedBuffer_ = NULL; 
      // todo : trouver une autre methode plus efficace que le polling et le sleep ! 
      usleep(1000);
    }
  }
}


int QSerialPort::nbBytesToRead()
{
    int bytes;

    ioctl(handlePort, FIONREAD, &bytes);
    
    return bytes;
}

//////////////////////////////////////////////////////////////////////////
/// Process the data received by the processIncomingEvent() function
/// It may be either bytes (data) or a ring indicator signal (PPS for GPS signal)
void QSerialPort::processIncomingData()
{
    // data frame
    if (numberBytesRead > 0) {
    FRAME * frame = new FRAME; 
    frame->t = t_; 
    frame->tr = 0; 
    frame->length = numberBytesRead; 
    frame->data = new char[frame->length]; 
    memcpy(frame->data, receivedBuffer_, frame->length); 
    
    frameLock_.lock(); 
    dataFrameList.append( frame );   
    frameLock_.unlock();

    //printf(receivedBuffer_); 
    emit newDataAvailable(1); 
    delete[] receivedBuffer_; 
    receivedBuffer_ = NULL;
  }

  if (ringIndicatorDetected) {
    ringIndicatorDetected = false;
    FRAME * frame = new FRAME; 
    frame->t = t_; 
    frame->tr = 0; 
    frame->length = 3; 
    frame->data = new char[frame->length]; 
    memcpy(frame->data, "PPS", frame->length); 
    frameLock_.lock(); 
    dataFrameList.append(frame); 
    frameLock_.unlock();

    emit newDataAvailable(1); 
  }

  // re-initialization
  t_ = 0;
  numberBytesToRead = 0; 
} 

/*int WifibotDPSIC::SendData()
{
    char buf[SIZE_MSG_SEND];

    if(m_changed)
        SetData(buf);


    if(mPort->write(buf,SIZE_MSG_SEND) == SIZE_MSG_SEND)
        return 1;
    else
        return 0;
}

void WifibotDPSIC::RecieveData()
{
    QByteArray buf;

    int a = mPort->bytesAvailable();
    buf.resize(1);
    do{
        mPort->read(buf.data(), 1);

    } while((uchar)buf.data()[0]!=0xFF);       // wait for first byte 0xFF

    do{
        a = mPort->bytesAvailable();
    } while(a<SIZE_MSG_RECEIVE);              // wait for a complete message (21 bytes)

    buf.resize(SIZE_MSG_RECEIVE);
    serialPort->read(buf.data(), SIZE_MSG_RECEIVE);   // read data

    short crcRead = ((short)buf.data()[20] << 8) +  (unsigned char)buf.data()[19];
    short crcComputed = CRC16((unsigned char*)buf.data(),19);

    // Check CRC
    if(crcRead != crcComputed)
    {
        qDebug() << "crc corrupted" << crcRead << "!=" << crcComputed ;
        return;
    }
    else
    {
        qDebug() << "data received" ;

        emit sendWifibotData();
        return;
    }

}
*/
