/**
 **                   Copyright 2015 by KVASER AB, SWEDEN
 **                        WWW: http://www.kvaser.com
 **
 ** This software is furnished under a license and may be used and copied
 ** only in accordance with the terms of such license.
 **
 ** \file kvmlib.h
 ** \brief   Library for accessing Kvaser Memorator (2nd generation)
 **
 ** Description:
 **         Library for accessing Kvaser Memorator (2nd generation). This
 **         library is used to extract log data, initialize disk, read and
 **         write configuration to a device, handle on device databases and
 **         more.
 **
 ** ---------------------------------------------------------------------------
 */


#ifndef KVMLIB_H
#define KVMLIB_H


#include <windows.h>
#include <stdio.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \name kvmDEVICE_xxx
 * \anchor kvmDEVICE_xxx
 *
 * Device type, used to connect to a Memorator device.
 *
 * @{
 */
#define kvmDEVICE_MHYDRA    0      ///< Kvaser Memorator (2nd generation)
/** @} */

/**
 * \brief A handle to a KME file
 */
typedef HANDLE kmeFileHandle;

/**
 * \name kvmFS_xxx
 * \anchor kvmFS_xxx
 *
 * File system used when formatting disk.
 *
 * @{
 */
#define kvmFS_FAT16   0      ///< fat16
#define kvmFS_FAT32   1      ///< fat32
/** @} */

/**
 * \name kvmFILE_xxx
 * \anchor kvmFILE_xxx
 *
 * KME file type, a binary file format representing log data.
 *
 * @{
 */
#define kvmFILE_KME24   0      ///< Deprecated
#define kvmFILE_KME25   1      ///< Deprecated
#define kvmFILE_KME40   2      ///< Kvaser binary format (KME 4.0)
/** @} */

/**
 * \brief A handle to a Memorator or equivalent KMF file
 */
typedef HANDLE kvmHandle;

typedef signed char                     int8;
typedef unsigned char                   uint8;
typedef short                           int16;
typedef unsigned short                  uint16;
typedef long int                        int32;
typedef unsigned long int               uint32;
typedef __int64                         int64;


/**
 * \name kvmStatus
 * \anchor kvmERR_xxx
 *
 * Generally, a return code greater than or equal to zero means success. A
 * value less than zero means failure.
 *
 * @{
 */

  // If you change here, remember to also change kvmlib.cpp and kvmlibtest...
typedef enum {
  kvmOK                      =  0,  ///< OK!
  kvmFail                    = -1,  ///< Generic error.
  kvmERR_PARAM               = -3,  ///< Error in supplied parameters.
  kvmERR_LOGFILEOPEN         = -8,  ///< Can't find/open log file.
  kvmERR_NOSTARTTIME         = -9,  ///< Start time not found.
  kvmERR_NOLOGMSG            = -10,  ///< No log message found.
  kvmERR_LOGFILEWRITE        = -11,  ///< Error writing log file.
  kvmEOF                     = -12,  ///< End of file found.
  kvmERR_NO_DISK             = -13,  ///< No disk found.
  kvmERR_LOGFILEREAD         = -14,  ///< Error while reading log file.
  kvmERR_QUEUE_FULL          = -20,  ///< Queue is full.
  kvmERR_CRC_ERROR           = -21,  ///< CRC check failed.
  kvmERR_SECTOR_ERASED       = -22,  ///< Sector unexpectadly erased.
  kvmERR_FILE_ERROR          = -23,  ///< File I/O error.
  kvmERR_DISK_ERROR          = -24,  ///< General disk error.
  kvmERR_DISKFULL_DIR        = -25,  ///< Disk full (directory).
  kvmERR_DISKFULL_DATA       = -26,  ///< Disk full (data).
  kvmERR_SEQ_ERROR           = -27,  ///< Unexpected sequence.
  kvmERR_FILE_SYSTEM_CORRUPT = -28,  ///< File system corrupt.
  kvmERR_UNSUPPORTED_VERSION = -29,  ///< Unsupported version.
  kvmERR_NOT_IMPLEMENTED     = -30,  ///< Not implemented.
  kvmERR_FATAL_ERROR         = -31,  ///< Fatal error.
  kvmERR_ILLEGAL_REQUEST     = -32,  ///< Illegal request.
  kvmERR_FILE_NOT_FOUND      = -33,  ///< File not found.
  kvmERR_NOT_FORMATTED       = -34,  ///< Disk not formatted.
  kvmERR_WRONG_DISK_TYPE     = -35,  ///< Wrong disk type.
  kvmERR_TIMEOUT             = -36,  ///< Timeout.
  kvmERR_DEVICE_COMM_ERROR   = -37,  ///< Device communication error.
  kvmERR_OCCUPIED            = -38,  ///< Device occupied.
  kvmERR_USER_CANCEL         = -39,  ///< User abort.
  kvmERR_FIRMWARE            = -40,  ///< Firmware error.
  kvmERR_CONFIG_ERROR        = -41,  ///< Configuration error.
  kvmERR_WRITE_PROT          = -42,  ///< Disk is write protected.

} kvmStatus;
/** @} */


/**
 * \name kvm_SWINFO_xxx
 * \anchor kvm_SWINFO_xxx
 *
 * Different types of version information that can be extracted using
 * kvmDeviceGetSoftwareInfo()
 *
 * @{
 */
#define kvm_SWINFO_DRIVER                   2  ///< Returns the used driver version information.
#define kvm_SWINFO_FIRMWARE                 3  ///< Returns the device firmware version information.
#define kvm_SWINFO_DRIVER_PRODUCT           4  ///< Obsolete. Returns the product version information.
#define kvm_SWINFO_CONFIG_VERSION_NEEDED    5  ///< Returns the version of the binary format (param.lif).
#define kvm_SWINFO_CPLD_VERSION             6  ///< Obsolete.
/** @} */

#include <pshpack1.h>

#ifndef canMSG_RTR
/**
 * \name canMSG_xxx
 *
 * The following flags can be found in a kvmLogMsgEx message flags field
 * returned from kvmKmeReadEvent(). All flags and/or combinations of them are
 * meaningful for logged message.
 * \anchor canMSG_xxx
 * @{
 */
#  define canMSG_RTR            0x0001      ///< Message is a remote request
#  define canMSG_STD            0x0002      ///< Message has a standard ID
#  define canMSG_EXT            0x0004      ///< Message has an extended ID
#  define canMSG_ERROR_FRAME    0x0020      ///< Message is an error frame
#  define canMSG_TXACK          0x0040      ///< Message is a TX ACK (msg is really sent)
#  define canMSG_TXRQ           0x0080      ///< Message is a TX REQUEST (msg is transfered to the CAN controller chip)
#  define canMSGERR_OVERRUN     0x0600      ///< Message overrun condition occurred.
/** @} */
#endif

/**
 * \name TRIGVAR_TYPE_xxx
 *
 * The following trigger types can be found in a kvmLogTriggerEx message type
 * field.
 * \anchor TRIGVAR_TYPE_xxx
 * @{
 */
#define TRIGVAR_TYPE_MSG_ID         0       ///< Message ID trigger
#define TRIGVAR_TYPE_MSG_DLC        1       ///< Message DLC trigger
#define TRIGVAR_TYPE_MSG_FLAG       2       ///< Message flag trigger
#define TRIGVAR_TYPE_SIGVAL         3       ///< Signal value trigger
#define TRIGVAR_TYPE_EXTERNAL       4       ///< External trigger
#define TRIGVAR_TYPE_TIMER          5       ///< Timer trigger
#define TRIGVAR_TYPE_DISK_FULL      6       ///< Disk is full trigger
#define TRIGVAR_TYPE_STARTUP        9       ///< Startup trigger
/** @} */

/**
 * \brief A CAN message
 */
typedef struct {
  uint32 id;            ///< The message identifier
  int64  timeStamp;     ///< The timestamp in units of 1 nanosecond
  uint32 channel;       ///< The device channel on which the message arrived, 0,1,...
  uint32 dlc;           ///< The length of the message
  uint32 flags;         ///< Message flags \ref canMSG_xxx
  uint8  data[64];      ///< Message data (64 bytes)
} kvmLogMsgEx;

/**
 * \brief A RTC clock message
 */
typedef struct {
  uint32 calendarTime;   ///< RTC date, seconds since 1970-01-01T00:00:00+00:00 (UTC)
  int64  timeStamp;      ///< The timestamp in units of 1 nanosecond
} kvmLogRtcClockEx;


/**
 * \brief A trigger message
 */
typedef struct {
  int32   type;           ///< The type of trigger \ref TRIGVAR_TYPE_xxx
  int32   preTrigger;     ///< Pretrigger time in milliseconds
  int32   postTrigger;    ///< Posttrigger time in milliseconds
  uint32  trigMask;       ///< Bitmask with all active triggers
  int64   timeStamp;      ///< The timestamp in units of 1 nanosecond
} kvmLogTriggerEx;

/**
 * \name kvmLOG_TYPE_xxx
 * \anchor kvmLOG_TYPE_xxx
 *
 * Event types in log
 * @{
 */
#define kvmLOG_TYPE_INVALID    0 ///< Invalid MEMOLOG type
#define kvmLOG_TYPE_CLOCK      1 ///< The type used in kvmLogRtcClockEx
#define kvmLOG_TYPE_MSG        2 ///< The type used in kvmLogMsgEx
#define kvmLOG_TYPE_TRIGGER    3 ///< The type used in kvmLogTriggerEx
/** @} */

/**
 * \brief The union of events used by kvmKmeReadEvent().
 */
typedef struct {
  uint32               type;    ///< \ref MEMOLOG_TYPE_xxx, Event types in log
  union {
    kvmLogMsgEx        msg;     ///< A CAN message
    kvmLogRtcClockEx   rtc;     ///< An RTC message
    kvmLogTriggerEx    trig;    ///< A trigger message
    uint8              raw[128]; ///< Raw data in a array
  } eventUnion;
} kvmLogEventEx;

#include <poppack.h>

/**
 * \ingroup Initialization
 *
 * This function must be called before any other functions are used.  It will
 * initialize the memolib library.
 */
void WINAPI kvmInitialize(void);

/**
 * \ingroup Initialization
 *
 * Convert a \ref kvmStatus errorcode to a text.
 *
 * \param[in]  error         The error code to convert.
 * \param[out] buf           Buffer to receive error text.
 * \param[in]  len           Buffer size in bytes.
 */
kvmStatus WINAPI kvmGetErrorText(kvmStatus error, char *buf, size_t len);

/**
 * \ingroup Connection
 *
 * Close the connection to the Memorator (device or file) opened with
 * kvmOpenDevice() or kvmDeviceMountKmf(). The handle becomes invalid.
 *
 * \param[in]  h         An open kvmHandle.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 *
 * \sa kvmDeviceMountKmf(), kvmDeviceOpen()
 */
kvmStatus WINAPI kvmClose(kvmHandle h);

/**
 *\ingroup Connection
 *
 * Connect to a Memorator device and obtain a handle for subsequent device operations.
 * The argument \a cardNr is the Card Number property (decreased by one) displayed in
 * Kvaser Hardware.
 *
 * \param[in]  cardNr      Card number
 * \param[out] status      \ref kvmOK if completely successful,
 *                         \ref kvmERR_xxx (negative) if failure
 * \param[in]  deviceType  \ref kvmDEVICE_xxx
 *
 * \return Returns an open handle to a Memorator on success.
 *
 * \sa kvmClose(), kvmLogFileMount()
 */
kvmHandle    WINAPI kvmDeviceOpen(int32 cardNr,
                                     kvmStatus *status,
                                     int32 deviceType);

/**
 *\ingroup Connection
 *
 * Mount the log area on the SD card on a connected Kvaser Memorator.
 *
 * \note Must be called after \ref kvmDeviceOpen before any subsequent
 * log operations are called.
 *
 * param[in]  h         An open kvmHandle.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 *
 * kvmClose(), kvmDeviceOpen()
 */
kvmStatus WINAPI kvmDeviceMountKmf(kvmHandle h);

/**
 * \ingroup Connection
 *
 * Open a KMF file on a hard disk or SD card reader and obtain a handle for
 * subsequent operations. \a deviceType is the device type that generated the
 * file.
 *
 * \param[in]  filename    KMF filename
 * \param[out] status      \ref kvmOK if successful, otherwise
 *                         \ref kvmERR_xxx
 * \param[in]  deviceType  \ref kvmDEVICE_xxx
 *
 * \return Returns an open handle to a Memorator on success.
 *
 * \sa kvmClose(), kvmDeviceOpen()
 *
 * \todo What is returned upon failure?
 */
kvmHandle    WINAPI kvmKmfOpen(const char *filename,
                                 kvmStatus *status,
                                 int32 deviceType);

/**
 * \ingroup DiskOperations
 *
 * Check for errors
 *
 * \param[in]  h         An open kvmHandle.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 */
kvmStatus WINAPI kvmKmfValidate(kvmHandle h);

/**
 * \ingroup DiskOperations
 *
 * Format the SD memory card in a connected Memorator.
 *
 *
 * \param[in]  h             An open kvmHandle.
 * \param[in]  fileSystem    Creates FAT32 if set to non-zero, else creates FAT16. \ref kvmFS_xxx
 * \param[in]  reserveSpace  Space to reserve for user files, in MB.
 * \param[in]  dbaseSpace    Space to reserve for database files, in MB.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 *
 * \sa kvmDeviceDiskSize
 */
kvmStatus WINAPI kvmDeviceFormatDisk(kvmHandle h,
                                   int fileSystem,
                                   uint32 reserveSpace,
                                   uint32 dbaseSpace);

/**
 * \ingroup DataExtraction
 *
 * Count the number of log files
 *
 * \param[in]  h             An open kvmHandle.
 * \param[out] fileCount     The number of log files on disk.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 */
kvmStatus WINAPI kvmLogFileGetCount(kvmHandle h, uint32* fileCount);

/**
 * \ingroup DataExtraction
 *
 * Mount the log file with the specified index. The index starts at 0. The
 * approximate number of events in the log file is returned.
 *
 * \param[in]  h             An open kvmHandle.
 * \param[in]  fileIndx      Index of the log file to open.
 * \param[out] eventCount    The approximate number of events in the log file
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 *
 * \sa kvmLogFileDismount()
 */
kvmStatus WINAPI kvmLogFileMount(kvmHandle h, uint32 fileIndx,
                                    uint32 *eventCount);

/**
 * \ingroup DataExtraction
 *
 * Dismount the log file opened with kvmLogFileMount(). The handle will stay valid.
 *
 * \param[in]  h             An open kvmHandle.
 */
kvmStatus WINAPI kvmLogFileDismount(kvmHandle h);

/**
 * \ingroup DataExtraction
 *
 * Get the time of the first event in the log file. The time is returned in
 * standard unix time format (number of seconds since 1970-01-01T00:00:00+00:00).
 *
 * \param[in]   h            An open kvmHandle.
 * \param[out]  startTime    The time of the first event in the log file (UTC)
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 */
kvmStatus WINAPI kvmLogFileGetStartTime (kvmHandle h,
                                         uint32 *startTime);

/**
 * \ingroup DataExtraction
 *
 * Get the serialnumber of the interface that created the log file.
 *
 * \param[in]   h              An open kvmHandle.
 * \param[out]  serialNumber   The serialnumber of the interface that created the
 *                             log file.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 */
kvmStatus WINAPI kvmLogFileGetCreatorSerial(kvmHandle h,
                                         uint32 *serialNumber);

/**
 * \ingroup DataExtraction
 *
 * Read an event from a log file opened with kvmLogFileMount().
 *
 * \param[in]   h              An open kvmHandle.
 * \param[out]  e              Event from log file.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 *
 * \todo Will a call to this function increment to next event?
 */
kvmStatus WINAPI kvmLogFileReadEvent(kvmHandle h, kvmLogEventEx *e);


/**
 * \ingroup DataExtraction
 *
 * Delete all log files from a Memorator.
 *
 * \param[in]  h         An open kvmHandle.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 */
kvmStatus WINAPI kvmLogFileDeleteAll(kvmHandle h);

/**
 * \ingroup SystemInformation
 *
 * Check if the SD memory card is present.
 *
 * \note This function is not supported by all devices.
 *
 * \param[in]   h              An open kvmHandle.
 * \param[out]  present        Non-zero means that SD memory card is present.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 */
kvmStatus WINAPI kvmDeviceDiskStatus(kvmHandle h, int *present);

/**
 * \ingroup SystemInformation
 *
 * Get disk usage statistics, reported in number of (512 byte) sectors.
 *
 *
 * \param[in]   h                 An open kvmHandle.
 * \param[out]  totalSectorCount  Total number of sectors devoted for logging
 * \param[out]  usedSectorCount   Number of logging sectors used
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 */
kvmStatus WINAPI kvmKmfGetUsage(kvmHandle h,
                                           uint32 *totalSectorCount,
                                           uint32 *usedSectorCount);

/**
 * \ingroup SystemInformation
 *
 * Get disk size, reported in number of (512 byte) sectors.
 *
 *
 * \param[in]   h              An open kvmHandle.
 * \param[out]  diskSize       Disk size in number of (512 byte) sectors.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 */
kvmStatus WINAPI kvmDeviceDiskSize(kvmHandle h, uint32 *diskSize);

/**
 * \ingroup SystemInformation
 *
 * Get serial number related to the Memorator handle.
 *
 * \param[in]   h            An open kvmHandle.
 * \param[out]  serial       Serial number of connected device.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 */
kvmStatus WINAPI kvmDeviceGetSerialNumber(kvmHandle h, unsigned int *serial);

/**
 * \ingroup SystemInformation
 *
 * Get software version information.
 *
 * \param[in]   h            An open kvmHandle.
 * \param[in]   itemCode     An item code specifying the type of version to get. \ref kvm_SWINFO_xxx
 * \param[out]  major        Major version number
 * \param[out]  minor        Minor version number
 * \param[out]  build        Build number
 * \param[out]  flags        For internal use only
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 */
kvmStatus WINAPI kvmDeviceGetSoftwareInfo(kvmHandle h,
                                               int32 itemCode,
                                               unsigned int *major,
                                               unsigned int *minor,
                                               unsigned int *build,
                                               unsigned int *flags);
/**
 * \ingroup SystemInformation
 *
 * Flash all LEDs on the opened Memorator device
 *
 * \param[in]   h            An open kvmHandle.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 */
kvmStatus WINAPI kvmDeviceFlashLeds(kvmHandle h);


/**
 * \ingroup RealTimeClock
 *
 * Get date and time from the RTC chip.  The time is returned in standard
 * unix time format (number of seconds since 1970-01-01T00:00:00+00:00).
 * Only for device handles
 *
 * \param[in]   h            An open kvmHandle.
 * \param[out]  t            Time in Unix time format
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 */
kvmStatus WINAPI kvmDeviceGetRTC(kvmHandle h, unsigned long *t);

/**
 * \ingroup RealTimeClock
 *
 * Set date and time in the RTC. The time is returned in standard unix
 * time format (number of seconds since 1970-01-01T00:00:00+00:00).
 * Only for device handles.
 *
 * \param[in]   h            An open kvmHandle.
 * \param[in]   t            Time in Unix time format
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 */
kvmStatus WINAPI kvmDeviceSetRTC(kvmHandle h, unsigned long t);

/**
 * \ingroup Configurations
 *
 * Read binary configuration data (param.lif) from a KMF file.
 *
 * \param[in]   h            An open kvmHandle.
 * \param[out]  buf          A pointer to buffer where the configuration (param.lif) will be written.
 * \param[in]   buflen       The length of the buffer buf.
 * \param[out]  actual_len   The actual length of the configuration written to buf.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 */
kvmStatus WINAPI kvmKmfReadConfig(kvmHandle h, void *buf,
                                    size_t buflen, size_t *actual_len);

/**
 * \ingroup Configurations
 *
 * Write binary configuration data (param.lif) to a KMF file.
 *
 * \param[in]   h            An open kvmHandle.
 * \param[in]   buf          A pointer to buffer containing the configuration (param.lif) to be written.
 * \param[in]   buflen       The length of the buffer buf.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 */
kvmStatus WINAPI kvmKmfWriteConfig(kvmHandle h, void *buf, size_t buflen);


/**
 * \ingroup Database
 *
 * Read the database file. The database will be extracted to path and the
 * name of the created file copied to filenamebuf.
 *
 * \param[in]   h            An open kvmHandle.
 * \param[in]   path         The path where the database file will be stored.
 * \param[out]  filenamebuf  The filename of the database. (should be greater then 12 bytes)
 * \param[in]   buflen       The lenght of filenamebuf
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 *
 * \sa kvmKmfPutDbaseFile()
 */
kvmStatus WINAPI kvmKmfGetDbaseFile(kvmHandle h, char *path, char *filenamebuf, size_t buflen);

/**
 * \ingroup Database
 *
 * Write the database file
 *
 * \param[in]   h            An open kvmHandle.
 * \param[in]   filename     The full path and name of the file, e.g. C:\\temp\\myfile.data
 *                           Note that the filename will be trucated to an 8.3 filename.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 *
 * \sa kvmKmfGetDbaseFile()
 */
kvmStatus WINAPI kvmKmfPutDbaseFile(kvmHandle h, char *filename);

/**
 * \ingroup Database
 *
 * Erase the database file
 *
 * \param[in]   h            An open kvmHandle.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 */
kvmStatus WINAPI kvmKmfEraseDbaseFile(kvmHandle h);

/**
 * \ingroup Files
 *
 * Open a KME file for reading and obtain a handle for subsequent operations.
 *
 * \param[in]   filename     The full path and name of the KME file, e.g. C:\\temp\\myfile.kme
 * \param[in]   fileType     \ref kvmFILE_xxx
 * \param[out]  status       \ref kvmOK (zero) if success
 *                           \ref kvmERR_xxx (negative) if failure
 *
 * \return Returns an open handle to a KME file on success.
 *
 * \sa kvmKmeReadEvent(), kvmKmeCountEvents(), kvmKmeCloseFile()
 *
 */
kmeFileHandle    WINAPI kvmKmeOpenFile (const char *filename,
                                    kvmStatus *status,
                                    int32 fileType);
/**
 * \ingroup Files
 *
 * Open a KME file for writing and obtain a handle for subsequent operations.
 * Note that kvmKmeCreateFile() will overwrite any existing file and that
 * \ref kvmFILE_KME24 and \ref kvmFILE_KME25 are deprecated formats. Please
 * use \ref kvmFILE_KME40.
 *
 * \param[in]   filename     The full path and name of the KME file, e.g. C:\\temp\\myfile.kme
 * \param[in]   fileType     \ref kvmFILE_xxx
 * \param[out]  status       \ref kvmOK (zero) if success
 *                           \ref kvmERR_xxx (negative) if failure
 *
 * \return Returns an open handle to a KME file on success.
 *
 * \sa kvmKmeWriteEvent(), kvmKmeCountEvents(), kvmKmeCloseFile()
 *
 */
kmeFileHandle    WINAPI kvmKmeCreateFile (const char *filename,
                                      kvmStatus *status,
                                      int32 fileType);

/**
 * \ingroup Files
 *
 * Read an event from a KME file opened with kvmKmeOpenFile().
 *
 * \param[in]   h              An open handle to a KME file.
 * \param[out]  e              Event from a KME file.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_NOLOGMSG on EOF
 * \return \ref kvmERR_xxx (negative) if failure
*
 * \sa kvmKmeOpenFile(), kvmKmeCountEvents(), kvmKmeCloseFile()
 *
 */
kvmStatus    WINAPI kvmKmeReadEvent (kmeFileHandle h, kvmLogEventEx *e);

/**
 * \ingroup Files
 *
 * Write an event to a KME file created with kvmKmeCreateFile().
 *
 * \param[in]  h               An open handle to a KME file.
 * \param[in]  e               Event to write.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
*
 * \sa kvmKmeCreateFile(), kvmKmeCountEvents(), kvmKmeCloseFile()
 *
 */
kvmStatus    WINAPI kvmKmeWriteEvent (kmeFileHandle h, kvmLogEventEx *e);

/**
 * \ingroup Files
 *
 * Count the number of events in a KME file.
 *
 * \param[in]  h               An open handle to a KME file.
 * \param[out] eventCount      Approximate number of events in a KME file.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 *
 * \sa kvmKmeOpenFile(), kvmKmeCreateFile()
 *
 */
kvmStatus    WINAPI kvmKmeCountEvents(kmeFileHandle h, uint32 *eventCount);

/**
 * \ingroup Files
 *
 * Close an open KME file opened with kvmKmeOpenFile() or created
 * with kvmKmeCreateFile(). The handle becomes invalid.
 *
 * \param[in]  h               An open handle to a KME file.
 *
 * \return \ref kvmOK (zero) if success
 * \return \ref kvmERR_xxx (negative) if failure
 *
 * \sa kvmKmeOpenFile(), kvmKmeCreateFile()
 *
 */
kvmStatus    WINAPI kvmKmeCloseFile (kmeFileHandle h);

#ifdef __cplusplus
}
#endif

#endif //KVMLIB_H
