/**
 * \section LICENSE
 *                   Copyright 2002-2011 by KVASER AB, SWEDEN
 *
 *                        WWW: http://www.kvaser.com
 *
 * This software is furnished under a license and may be used and copied
 * only in accordance with the terms of such license.
 *
 * \section DESCRIPTION
 *
 *   Definitions for the CAN Database API.
 *
 * \file kvaDBlib.h
 * \author Kvaser AB
 *
 */

#ifndef KVADBLIB_H
#define KVADBLIB_H

/**
 * \defgroup CANDB  CAN Database
 * @{
 */


#ifdef __cplusplus
extern "C" {
#endif


/**
 *
 * \name kvaDbErr_xxx
 * \anchor kvaDbErr_xxx
 *
 * The following codes are returned by most CANDB API calls.
 *
 * \note Error codes are always negative.
 * @{
 */
typedef enum {
    kvaDbOK                      =  0,  ///< OK - no error
    kvaDbErr_Fail                = -1,  ///< General failure
    kvaDbErr_NoDatabase          = -2,  ///< No database was found
    kvaDbErr_Param               = -3,  ///< One or more of the parameters in call is erronous
    kvaDbErr_NoMsg               = -4,  ///< No message was found
    kvaDbErr_NoSignal            = -5,  ///< No signal was found
    kvaDbErr_Internal            = -6,  ///< An internal error occured in the library
    kvaDbErr_DbFileOpen          = -7,  ///< Could not open the database file
    kvaDbErr_DatabaseInternal    = -8,  ///< An internal error occured in the database handler
    kvaDbErr_NoNode              = -9,  ///< Could not find the database node
    kvaDbErr_NoAttrib            = -10, ///< No attribute found
    kvaDbErr_OnlyOneAllowed      = -11, ///< Only one CANdb structure is allowed
} KvaDbStatus;
/** @} */

/**
 * \name Kvaser Database Message Flags
 * \anchor KVADB_MESSAGE_xxx
 *
 * The following flags are never used.
 *
 * \todo Why are they never used?
 * @{
 */
#define KVADB_MESSAGE_EXT      0x80000000  ///< Message is an extended CAN message
#define KVADB_MESSAGE_J1939    0x00000001  ///< Message uses J1939 protocol
#define KVADB_MESSAGE_WAKEUP   0x00000002  ///< Message is a wake-up frame, Id/msg/dlc are undefined
/** @} */

/**
 * \name CANDBSignalEncoding API Signal Encoding
 * 
 * The following values are used by kvaDbGetSignalEncoding() and
 * kvaDbSetSignalEncoding()
 * @{
 */
typedef enum {
  kvaDb_Intel     = 0, ///< Intel
  kvaDb_Motorola  = 1, ///< kvaDb_Motorola
} KvaDbSignalEncoding;
/** @} */

/**
 * \name CANDBSignalRepresentation API Signal Representation
 * 
 * The type ::KvaDbSignalType is used by e.g. kvaDbGetSignalRepresentationType().
 * @{
 */
typedef enum  {
   kvaDb_Invalid,   ///< Invalid representation
   kvaDb_Signed,    ///< Signed integer
   kvaDb_Unsigned,  ///< Unsigned integer
   kvaDb_Float,     ///< Float
   kvaDb_Double     ///< Double
} KvaDbSignalType;
/** @} */

/**
 * \name CANDBAttributeRepresentation API Attribute Representation
 *
 * The type ::KvaDbAttributeType is used by e.g. kvaDbGetAttributeDefinitionType()
 * and kvaDbGetAttributeType().
 * @{
 */
typedef enum {
   kvaDb_AttributeInvalid       = 0,  ///< Invalid attribute
   kvaDb_AttributeInteger       = 1,  ///< Integer attribute
   kvaDb_AttributeFloat         = 3,  ///< Float attribute
   kvaDb_AttributeEnumeration   = 4,  ///< Enumeration attribute
   kvaDb_AttributeString        = 5,  ///< String attribute
} KvaDbAttributeType;
/** @} */

/**
 * \name CANDBAttributeOwner API Attribute Owner
 *
 * The type ::KvaDbAttributeOwnerType is used by e.g. kvaDbGetAttributeDefinitionOwner(),
 * @{
 */
typedef enum {
   kvaDb_AttributeOwnerInvalid  = 0,  ///< Invalid owner
   kvaDb_AttributeOwnerDb       = 1,  ///< Database owner
   kvaDb_AttributeOwnerMessage  = 2,  ///< Message owner
   kvaDb_AttributeOwnerNode     = 3,  ///< Node owner
   kvaDb_AttributeOwnerSignal   = 4,  ///< Signal owner
   kvaDb_AttributeOwnerEnv      = 5,  ///< Environment owner
} KvaDbAttributeOwner;
/** @} */

// Database flags

/**
 * \name CANDBDatabaseFlags API Database Flags
 * \anchor KVADB_DATABASE_xxx
 * 
 * The following value is used by kvaDbGetFlags().
 * @{
 */
#define KVADB_DATABASE_J1939   0x0001  ///< The database uses the J1939 protocol

typedef void* KvaDbHnd;              ///< Database handle
typedef void* KvaDbMessageHnd;       ///< Message handle
typedef void* KvaDbSignalHnd;        ///< Signal handle
typedef void* KvaDbNodeHnd;          ///< Node handle
typedef void* KvaDbAttributeHnd;     ///< Attribute handle
typedef void* KvaDbAttributeDefHnd;  ///< Attribute definition handle
/** @} */

/**
 * \name Kvaser Database Multiplexer Mode values
 * \anchor KVADB_MUX_xxx
 * @{ 
*/
#define KVADB_MUX_SIGNAL -2  ///< Multiplex mode value of a multiplexer signal
#define KVADB_MUX_INDEPENDENT -1 ///< Miltiplex mode value of an independent signal
/** @} */

/**
 * This function creates a new database handle.
 * 
 * \param[out] dh  A handle to a new database
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbClose()
 */
KvaDbStatus WINAPI kvaDbOpen(KvaDbHnd *dh);

/**
 * This loads a database file into a handle created with kvaDbOpen().
 * 
 * \note This function is deprecated and will be replaced by kvaDbCreate()
 *
 * \param[in] dh        A handle database handle
 * \param[in] filename  A database file
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbCreate()
 */
KvaDbStatus WINAPI kvaDbAddFile(KvaDbHnd dh, const char *filename);

/**
 * This function creates a new database in a handle created with
 * kvaDbOpen(). It is used when there is no file to read data from.
 *
 * \note This function is deprecated and will be replaced by kvaDbCreate().
 * 
 * \param[in] dh        A database handle
 * \param[in] filename  The the dummy filename to use for this database
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbCreate()
 */
KvaDbStatus WINAPI kvaDbSetDummyFileName(KvaDbHnd dh, const char *filename);

/**
 * This function creates a new database in a handle created with
 * kvaDbOpen(). There are three ways to call this function:
 *   
 * \li To load data from an existing database file, set \a localName to \c NULL
 * and set \a filename to the database file.
 *   
 * \li To add an empty database, set \a localName to any name and set \a
 * filename to \c NULL.
 *   
 * \li To load data from an existing database file and give it a new name, set
 * \a localName to the new name and set \a filename to the database file.
 * 
 * \param[in] dh         A database handle
 * \param[in] localName  The name of the database or \c NULL
 * \param[in] filename   The database file or \c NULL
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbOpen()
 */
KvaDbStatus WINAPI kvaDbCreate(KvaDbHnd dh, const char *localName, const char *filename);

/**
 * This function closes an open database handle.
 * 
 * \param[in] dh  A handle to a database
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbOpen()
 */
KvaDbStatus WINAPI kvaDbClose(KvaDbHnd dh);

/**
 * Write a database to file.
 * 
 * \param[in] dh        A handle to a database
 * \param[in] filename  The name of the database file
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbReadFile()
 */
KvaDbStatus WINAPI kvaDbWriteFile(KvaDbHnd dh, char* filename);

/**
 * Load a database from file into a handle created with kvaDbOpen().
 * 
 * \param[out]  dh        A handle to a database
 * \param[in]   filename  The name of the database file.
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbCreate()
 */
KvaDbStatus WINAPI kvaDbReadFile(KvaDbHnd dh, char* filename);

/**
 * Get the database flags.
 * 
 * \param[in]  dh     A handle to a database
 * \param[out] flags  The database flags, \ref KVADB_DATABASE_xxx
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbReadFile()
 *
 * \todo check kvaDbReadFile(), we should perhaps link to kvaDbCreate()?
 */
KvaDbStatus WINAPI kvaDbGetFlags(KvaDbHnd dh, unsigned int *flags);

/**
 * Get the first message in a database.
 * 
 * \param[in]  dh  A handle to a database
 * \param[out] mh  A handle to the first message
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetNextMsg()
 */
KvaDbStatus WINAPI kvaDbGetFirstMsg(KvaDbHnd dh, KvaDbMessageHnd *mh);

/**
 * Get a handle to the next message in a database. Should be called after
 * kvaDbGetFirstMsg().
 * 
 * \param[in]  dh  A database handle
 * \param[out] mh  A message handle
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetFirstMsg()
 */
KvaDbStatus WINAPI kvaDbGetNextMsg(KvaDbHnd dh, KvaDbMessageHnd *mh);

/**
 * Get a handle to a message with a specific identifier.
 * 
 * \param[in]  dh  A database handle
 * \param[in]  id  The selected message identifier
 * \param[out] mh  A message handle
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetMsgByName()
 */
KvaDbStatus WINAPI kvaDbGetMsgById(KvaDbHnd dh, 
                                   unsigned int id, 
                                   KvaDbMessageHnd *mh);

/**
 * Get a handle to a message with a specific name.
 * 
 * \param[in]  dh        A databse handle
 * \param[in]  msg_name  The message name to search for
 * \param[out] mh        A message handle
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetMsgById()
 */
KvaDbStatus WINAPI kvaDbGetMsgByName(KvaDbHnd dh, 
                                     const char *msg_name, 
                                     KvaDbMessageHnd *mh);

/**
 * Get the message name.
 * 
 * \param[in]  mh      A message handle
 * \param[out] buf     The buffer that will hold the message name
 * \param[in]  buflen  The length of the buffer.
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbSetMsgName(), kvaDbGetMsgQualifiedName()
 */
KvaDbStatus WINAPI kvaDbGetMsgName(KvaDbMessageHnd mh, char *buf, size_t buflen);

/**
 * Get the qualified message name, which is the database name and the message
 * name separated by a dot.
 * 
 * \param[in]  mh      A message handle
 * \param[out] buf     The buffer that will hold the qualified message name
 * \param[in]  buflen  The length of the buffer
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetMsgName()
 */
KvaDbStatus WINAPI kvaDbGetMsgQualifiedName(KvaDbMessageHnd mh, char *buf, size_t buflen);

/**
 * Get the message comment.
 * 
 * \param[in]  mh      A message handle
 * \param[out] buf     The buffer that will hold the message comment
 * \param[in]  buflen  The length of the buffer
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbSetMsgComment()
 */
KvaDbStatus WINAPI kvaDbGetMsgComment(KvaDbMessageHnd mh, char *buf, size_t buflen);

/**
 * Get message identifier and flags.
 * 
 * \param[in]  mh     A message handle.
 * \param[out] id     The identifier.
 * \param[out] flags  The flags, \ref KVADB_MESSAGE_xxx
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \todo "flags" contains extended? or use 0x80000000 convention?
 *       could also contain J1939 flag, wakeup etc. qqq
 *
 * \sa kvaDbSetMsgId()
 */
KvaDbStatus WINAPI kvaDbGetMsgId(KvaDbMessageHnd mh, 
                                 unsigned int *id, 
                                 unsigned int *flags);

/**
 * Get the data length code for a message.
 * 
 * \param[in]  mh   A message handle
 * \param[out] dlc  The data length code
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbSetMsgDlc()
 */
KvaDbStatus WINAPI kvaDbGetMsgDlc(KvaDbMessageHnd mh, int *dlc);

/**
 * Add a new message to a database.
 * 
 * \param[in]  dh  A database handle
 * \param[out] mh  A handle to the new message
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbDeleteMsg()
 */
KvaDbStatus WINAPI kvaDbAddMsg(KvaDbHnd dh, KvaDbMessageHnd *mh);

/**
 * Delete a message from a database.
 * 
 * \param[in] dh  A database handle
 * \param[in] mh  A handle to the message that will be deleted
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbAddMsg()
 */
KvaDbStatus WINAPI kvaDbDeleteMsg(KvaDbHnd dh, KvaDbMessageHnd mh);

/**
 * Set the database flags.
 *
 * \note This feature is currently not enabled.
 * 
 * \param[in] dh     A database handle
 * \param[in] flags  The flags to set, \ref KVADB_MESSAGE_xxx
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetFlags()
 */
KvaDbStatus WINAPI kvaDbSetFlags(KvaDbHnd dh, unsigned int flags);

/**
 * Set the message name.
 * 
 * \param[in] mh   A message handle
 * \param[in] buf  The buffer that contains the message name.
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetMsgName()
 */
KvaDbStatus WINAPI kvaDbSetMsgName(KvaDbMessageHnd mh, char *buf);

/**
 * Set the message comment.
 * 
 * \param[in] mh   A message handle
 * \param[in] buf The buffer that contains the message comment
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetMsgComment()
 */
KvaDbStatus WINAPI kvaDbSetMsgComment(KvaDbMessageHnd mh, char *buf);

/**
 * Set the message identifier and flags.
 * 
 * \param[in] mh     A message handle
 * \param[in] id     The message identifier
 * \param[in] flags  The message flags, \ref KVADB_MESSAGE_xxx
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \todo "flags" contains extended? or use 0x80000000 convention?
 *       could also contain J1939 flag, wakeup etc. qqq
 *
 * \sa kvaDbGetMsgId()
 */
KvaDbStatus WINAPI kvaDbSetMsgId(KvaDbMessageHnd mh, unsigned int id, unsigned int flags);

/**
 * Set the data length code for a message.
 * 
 * \param[in] mh   A message handle
 * \param[in] dlc  The data length code
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetMsgDlc()
 */
KvaDbStatus WINAPI kvaDbSetMsgDlc(KvaDbMessageHnd mh, int dlc);

// qqq pragma fr att stta sizeof(enum) till 4? annars skippa enum...

// J1939 fungerar hur? Helt enkelt genom att id == PGN?  Hur hantera
// SA, DA, etc? qqq

/**
 * Returns handle to a multiplexer signal of a given message
 *
 * \param[in]  mh     A message handle
 * \param[out] shh    Multiplexer signal handle
 *
 * \return ::kvaDbOK (zero if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 */ 

KvaDbStatus WINAPI kvaDbGetMsgMux(KvaDbMessageHnd mh, KvaDbSignalHnd *shh); 

/**
 * Get the first signal in a message.
 * 
 * \param[in]  mh  A message handle
 * \param[out] sh  A handle to the first signal
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetNextSignal()
 */
 
KvaDbStatus WINAPI kvaDbGetFirstSignal(KvaDbMessageHnd mh, KvaDbSignalHnd *sh);

/**
 * Get the next signal in a message. Should be called after
 * kvaDbGetFirstSignal().
 * 
 * \param[in]  mh  A message handle
 * \param[out] sh  A handle to the next signal
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetFirstSignal
 */
KvaDbStatus WINAPI kvaDbGetNextSignal(KvaDbMessageHnd mh, KvaDbSignalHnd *sh);

/**
 * Get a signal handle by seaching for the signals's name.
 * 
 * \param[in]  mh   A message handle
 * \param[in]  buf  The buffer that contains the name of the signal to search for
 * \param[out] sh   A handle to the signal
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetFirstMsg()
 */
KvaDbStatus WINAPI kvaDbGetSignalByName(KvaDbMessageHnd mh, char *buf, KvaDbSignalHnd *sh);

/**
 * Sets signal's multiplex mode. If value is equal to KVADB_MUX_SIGNAL, marks signal as a multiplexer.
 *
 * \param[in]  sh     A signal handle
 * \param[in] mux   Signal's multiplex mode
 *
 * \return ::kvaDbOK (zero if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * \sa \ref KVADB_MUX_xxx
 */ 
  
KvaDbStatus WINAPI kvaDbSetSignalMode(KvaDbSignalHnd sh, int mux);

/**
 * Sets mux to multiplexer mode value of given signal. 
 * See \ref KVADB_MUX_xxx for special (negative) values of multiplexer mode.
 *
 * \param[in]  sh     A signal handle
 * \param[out] mux   Signal's multiplex mode
 *
 * \return ::kvaDbOK (zero if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * \sa \ref KVADB_MUX_xxx
*/ 

KvaDbStatus WINAPI kvaDbGetSignalMode(KvaDbSignalHnd sh, int *mux);

/**
 * Convert data to a signal's physical (floating-point) value.
 * 
 * \param[in]  sh    A signal handle
 * \param[out] f     The converted value
 * \param[in]  data  The data buffer to be converted
 * \param[in]  len   The length of the data buffer
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbStoreSignalValuePhys()
 */

KvaDbStatus WINAPI kvaDbGetSignalValueFloat(KvaDbSignalHnd sh, double *f, void *data, size_t len);

/**
 * Convert data to a signal's raw (integer) value.
 * 
 * \param[in]  sh    A signal handle
 * \param[out] f     The converted value
 * \param[in]  data  The data buffer to be converted
 * \param[in]  len   The length of the data buffer
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbStoreSignalValueRaw()
 */
 
KvaDbStatus WINAPI kvaDbGetSignalValueInteger(KvaDbSignalHnd sh, int *f, void *data, size_t len);

/**
 * Get the minimum and maximum values from a signal.
 * 
 * \param[in]  sh      A signal handle
 * \param[out] minval  The minimum value
 * \param[out] maxval  The maximum value
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbSetSignalValueLimits()
 */
KvaDbStatus WINAPI kvaDbGetSignalValueLimits(KvaDbSignalHnd sh, double *minval, double *maxval);

/**
 * Get the scale factor and offset from a signal.
 * 
 * \param[in]  sh      A signal handle
 * \param[out] factor  The signal scale factor
 * \param[out] offset  The signal offset
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbSetSignalValueScaling()
 */
KvaDbStatus WINAPI kvaDbGetSignalValueScaling(KvaDbSignalHnd sh, double *factor, double *offset);

/**
 * Get the size and position of the signal value.
 * 
 * \param[in]  sh        A signal handle
 * \param[out] startbit  The bit where the signal value starts
 * \param[out] length    The length of the signal value in bits
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbSetSignalValueSize()
 */
KvaDbStatus WINAPI kvaDbGetSignalValueSize(KvaDbSignalHnd sh, int *startbit, int *length);

/**
 * Get the signal name.
 * 
 * \param[in]  sh      A signal handle
 * \param[out] buf     The signal name
 * \param[in]  buflen  The length of the buffer that will hold the signal name
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbSetSignalName()
 */
KvaDbStatus WINAPI kvaDbGetSignalName(KvaDbSignalHnd sh, char *buf, size_t buflen);

/**
 * Get the qualified signal name. which is the database, message ans signal
 * names separated by dots.
 * 
 * \param[in]  sh      A signal handle
 * \param[out] buf     The qualified signal name
 * \param[in]  buflen  The length of the buffer that will hold the qualified
 *                     signal name
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetSignalName()
 */
KvaDbStatus WINAPI kvaDbGetSignalQualifiedName(KvaDbSignalHnd sh, char *buf, size_t buflen);

/**
 * Get the signal comment.
 * 
 * \param[in]  sh      A signal handle
 * \param[out] buf     The signal comment
 * \param[in]  buflen  The length of the buffer that will hold the signal comment
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbSetSignalComment()
 */
KvaDbStatus WINAPI kvaDbGetSignalComment(KvaDbSignalHnd sh, char *buf, size_t buflen);

/**
 * Get the signal unit.
 * 
 * \param[in]  sh      A signal handle
 * \param[out] buf     The signal unit
 * \param[in]  buflen  The length of the buffer that will hold the signal unit
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbSetSignalUnit()
 */
KvaDbStatus WINAPI kvaDbGetSignalUnit(KvaDbSignalHnd sh, char *buf, size_t buflen);

/**
 * Get the signal encoding.
 * 
 * \param[in]  sh  A signal handle
 * \param[out] e   The signal encoding, Intel or Motorola, ::KvaDbSignalEncoding
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbSetSignalEncoding()
 */
KvaDbStatus WINAPI kvaDbGetSignalEncoding(KvaDbSignalHnd sh, KvaDbSignalEncoding *e);

/**
 * Get the signal representation type.
 * 
 * \param[in]  sh  A signal handle
 * \param[out] t   The signal representation type, ::KvaDbSignalType
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbSetSignalRepresentationType()
 */
KvaDbStatus WINAPI kvaDbGetSignalRepresentationType(KvaDbSignalHnd sh, KvaDbSignalType *t);

/**
 * Get the signal representation type.
 * 
 * \note This function is deprecated and will be replaced by
 * kvaDbGetSignalRepresentationType()
 * 
 * \param[in]  sh
 * \param[out] t   The signal representation type, ::KvaDbSignalType
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbSetSignalRepresentationType()
 */
KvaDbStatus WINAPI kvaDbGetSignalPresentationType(KvaDbSignalHnd sh, KvaDbSignalType *t);

/**
 * Add a new signal to a message.
 * 
 * \param[in]  mh  A message handle
 * \param[out] sh  A handle to the new signal
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbDeleteSignal()
 */
KvaDbStatus WINAPI kvaDbAddSignal(KvaDbMessageHnd mh, KvaDbSignalHnd *sh);

/**
 * Delete a signal from a message.
 * 
 * \param[in] mh  A message handle
 * \param[in] sh  A handle to the signal to delete
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbAddSignal()
 */
KvaDbStatus WINAPI kvaDbDeleteSignal(KvaDbMessageHnd mh, KvaDbSignalHnd sh);

/**
 * Set the value limits for a signal.
 * 
 * \param[in] sh      A signal handle
 * \param[in] minval  The minimum signal value
 * \param[in] maxval  The maximum signal value
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetSignalValueLimits()
 */
KvaDbStatus WINAPI kvaDbSetSignalValueLimits(KvaDbSignalHnd sh, 
                                             double minval, 
                                             double maxval);

/**
 * Set the scale factor and offset for a signal.
 * 
 * \param[in] sh      A signal handle
 * \param[in] factor  The scale factor for the signal value
 * \param[in] offset  The offset for the signal value
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetSignalValueScaling()
 */
KvaDbStatus WINAPI kvaDbSetSignalValueScaling(KvaDbSignalHnd sh, 
                                              double factor, 
                                              double offset);

/**
 * Set the value position and size for a signal.
 * 
 * \param[in] sh        A signal handle
 * \param[in] startbit  The starting bit of the signal's value
 * \param[in] length    The length of the value in bits
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetSignalValueSize()
 */
KvaDbStatus WINAPI kvaDbSetSignalValueSize(KvaDbSignalHnd sh, 
                                           int startbit, 
                                           int length);

/**
 * Set the signal name.
 * 
 * \param[in] sh   A signal handle
 * \param[in] buf  The buffer that contains the signal name
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetSignalName()
 */
KvaDbStatus WINAPI kvaDbSetSignalName(KvaDbSignalHnd sh, char *buf);

/**
 * Set the signal comment.
 * 
 * \param[in] sh   A signal handle
 * \param[in] buf  The buffer that contains the signal comment
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetSignalComment()
 */
KvaDbStatus WINAPI kvaDbSetSignalComment(KvaDbSignalHnd sh, char *buf);

/**
 * Set the signal unit.
 * 
 * \param[in] sh   A signal handle
 * \param[in] buf  The buffer that contains the unit for the signal
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetSignalUnit()
 */
KvaDbStatus WINAPI kvaDbSetSignalUnit(KvaDbSignalHnd sh, char *buf);

/**
 * Set the signal encoding.
 * 
 * \param[in] sh  A signal handle
 * \param[in] e   The signal encoding, Intel or Motorola, ::KvaDbSignalEncoding
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetSignalEncoding()
 */
KvaDbStatus WINAPI kvaDbSetSignalEncoding(KvaDbSignalHnd sh, KvaDbSignalEncoding e);

/**
 * Set the signal representation type.
 * 
 * \param[in] sh  A signal handle
 * \param[in] t   The signal representation type, ::KvaDbSignalType
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetSignalRepresentationType()
 */
KvaDbStatus WINAPI kvaDbSetSignalRepresentationType(KvaDbSignalHnd sh, KvaDbSignalType t);

/**
 * Get the first node.
 * 
 * \param[in]  dh  A database handle
 * \param[out] nh  A node handle
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetNextNode()
 */
KvaDbStatus WINAPI kvaDbGetFirstNode(KvaDbHnd dh, KvaDbNodeHnd *nh);

/**
 * Get the next node. Should be called after kvaDbGetFirstNode().
 * 
 * \param[in]  dh  A database handle
 * \param[out] nh  A handle to the next node
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetNodeByName()
 */
KvaDbStatus WINAPI kvaDbGetNextNode(KvaDbHnd dh, KvaDbNodeHnd *nh);

/**
 * Get a node by seaching for the node's name.
 * 
 * \param[in]  dh         A database handle
 * \param[in]  node_name  The buffer that contains the node name to search for
 * \param[out] nh         A node handle
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetNodeAttributeIntByName()
 */
KvaDbStatus WINAPI kvaDbGetNodeByName(KvaDbHnd dh, const char *node_name, KvaDbNodeHnd *nh);

/**
 * Get the value of a node attribute as an integer.
 * 
 * \param[in]  nh        A node handle
 * \param[in]  attrName  The buffer that contains the attribute name
 * \param[out] val       The value of the attribute
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetFirstNode()
 */
KvaDbStatus WINAPI kvaDbGetNodeAttributeIntByName(KvaDbNodeHnd nh, const char *attrName, int *val);

/**
 * Add a new node to a database.
 * 
 * \param[in]  dh  A database handle
 * \param[out] nh  A handle to the new node
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 */
KvaDbStatus WINAPI kvaDbAddNode(KvaDbHnd dh, KvaDbNodeHnd *nh);

/**
 * Delete a node from a database.
 * 
 * \param[in] dh  A database handle
 * \param[in] nh  A handle to the node that will be deleted
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 */
KvaDbStatus WINAPI kvaDbDeleteNode(KvaDbHnd dh, KvaDbNodeHnd nh);

/**
 * Set the node name.
 * 
 * \param[in] nh   A node handle
 * \param[in] buf  The buffer that contains the node name
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 */
KvaDbStatus WINAPI kvaDbSetNodeName(KvaDbNodeHnd nh, char *buf);

/**
 * Get the node name.
 * 
 * \param[in] nh   A node handle
 * \param[out] buf  The buffer that will contain the node name
 * \param[in] buflen  The length of the data buffer
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 */
KvaDbStatus WINAPI kvaDbGetNodeName(KvaDbNodeHnd nh, char *buf, size_t buflen);

/**
 * Set the node comment.
 * 
 * \param[in] nh   A node handle
 * \param[in] buf  The buffer that contains the node comment
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 */
KvaDbStatus WINAPI kvaDbSetNodeComment(KvaDbNodeHnd nh, char *buf);

/**
 * Add a receiving node to a signal.
 * 
 * \param[in] sh   A signal handle
 * \param[in] nh   A node handle
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 */
KvaDbStatus WINAPI kvaDbAddReceiveNodeToSignal(KvaDbSignalHnd sh, KvaDbNodeHnd nh);

/**
 * Remove a receiving node from a signal.
 * 
 * \param[in] sh   A signal handle
 * \param[in] nh   A node handle
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 */
KvaDbStatus WINAPI kvaDbRemoveReceiveNodeFromSignal(KvaDbSignalHnd sh, KvaDbNodeHnd nh);

/**
 * Convert a physical signal value to data bytes.
 * 
 * \param[in]  sh        A signal handle
 * \param[out] can_data  The buffer that will hold the converted value
 * \param[in]  dlc       The length of the data buffer
 * \param[in]  value     The physical signal value to convert
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetSignalValueFloat()
 */
KvaDbStatus WINAPI kvaDbStoreSignalValuePhys(KvaDbSignalHnd sh, unsigned char *can_data, int dlc, double value);

/**
 * Convert a raw signal value to data bytes.
 * 
 * \param[in]  sh        A signal handle
 * \param[out] can_data  The buffer that will hold the converted value
 * \param[in]  dlc       The length of the data buffer
 * \param[in]  value     The raw signal value to convert
 * 
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 * 
 * \sa kvaDbGetSignalValueInteger()
 */
KvaDbStatus WINAPI kvaDbStoreSignalValueRaw(KvaDbSignalHnd sh, unsigned char *can_data, int dlc, int value);

/**
 * Get a handle to the first attribute defined in the database.
 *
 * \param[in]  dh   A handle to a database
 * \param[out] adh  An attribute definition handle
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbGetNextAttributeDefinition()
 */
KvaDbStatus WINAPI kvaDbGetFirstAttributeDefinition(KvaDbHnd dh, KvaDbAttributeDefHnd *adh);

/**
 * Get a handle to an attribute definition with specified name.
 *
 * \param[in]  dh   A handle to a database
 * \param[in]  attrName  The attribute name to search for
 * \param[out] adh  An attribute definition handle
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 */
KvaDbStatus WINAPI kvaDbGetAttributeDefinitionByName(KvaDbHnd dh, const char *attrName, KvaDbAttributeDefHnd *adh);

/**
 * Get a handle to the following attribute defined.
 *
 * \param[in]  adh   A handle to an attribute definition database
 * \param[out] nadh  An attribute definition handle
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbGetFirstAttributeDefinition()
 */
KvaDbStatus WINAPI kvaDbGetNextAttributeDefinition(KvaDbAttributeDefHnd adh, KvaDbAttributeDefHnd *nadh);

/**
 * Get the type for an attribute definition.
 *
 * \param[in]  adh   A handle to an attribute definition
 * \param[out] at    The type of the attribute definition
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbGetAttributeDefinitionOwner()
 */
KvaDbStatus WINAPI kvaDbGetAttributeDefinitionType(KvaDbAttributeDefHnd adh, KvaDbAttributeType *at);

/**
 * Get the type for an attribute definition.
 *
 * \param[in]  adh   A handle to an attribute definition
 * \param[out] ao    The owner of the attribute definition
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbGetAttributeDefinitionType()
 */
KvaDbStatus WINAPI kvaDbGetAttributeDefinitionOwner(KvaDbAttributeDefHnd adh, KvaDbAttributeOwner *ao);

/**
 * Get the name for an attribute definition.
 *
 * \param[in]  adh     A handle to an attribute
 * \param[out] buf     The buffer that will hold the attribute name
 * \param[in]  buflen  The length of the buffer.
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbGetAttributeDefinitionType()
 */
KvaDbStatus WINAPI kvaDbGetAttributeDefinitionName(KvaDbAttributeDefHnd adh, char *buf, size_t buflen);

/**
 * Get the values for an attribute definition of type integer.
 *
 * \param[in]  adh   A handle to an attribute definition
 * \param[out] def   The default value of the attribute definition
 * \param[out] min   The min value of the attribute definition
 * \param[out] max   The max value of the attribute definition
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 */
KvaDbStatus WINAPI kvaDbGetAttributeDefinitionInt(KvaDbAttributeDefHnd adh, int *def, int *min, int *max);

/**
 * Get the values for an attribute definition of type float.
 *
 * \param[in]  adh   A handle to an attribute definition
 * \param[out] def   The default value of the attribute definition
 * \param[out] min   The min value of the attribute definition
 * \param[out] max   The max value of the attribute definition
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 */
KvaDbStatus WINAPI kvaDbGetAttributeDefinitionFloat(KvaDbAttributeDefHnd adh, float *def, float *min, float *max);

/**
 * Get the value for an attribute definition of type string.
 *
 * \param[in]  adh     A handle to an attribute definition
 * \param[out] buf     The buffer that holds the default value of the attribute
 * \param[out] buflen  The length of the buffer.
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 */
KvaDbStatus WINAPI kvaDbGetAttributeDefinitionString(KvaDbAttributeDefHnd adh, char *buf, int buflen);

/**
 * Get the value for an attribute definition of type enumeration.
 *
 * \param[in]  adh   A handle to an attribute definition
 * \param[out] def   The default value of the attribute definition
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 */
KvaDbStatus WINAPI kvaDbGetAttributeDefinitionEnumeration(KvaDbAttributeDefHnd adh, int *def);

/**
 * Add an attribute to a message.
 *
 * \param[in]  mh    A handle to a message
 * \param[in]  adh   A handle to an attribute definition
 * \param[out] ah    An attribute handle
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbDeleteMsgAttribute()
 */
KvaDbStatus WINAPI kvaDbAddMsgAttribute(KvaDbMessageHnd mh, KvaDbAttributeDefHnd adh, KvaDbAttributeHnd *ah);

/**
 * Delete an attribute from a message.
 *
 * \param[in]  mh    A handle to a message
 * \param[in]  ah    A handle to an attribute
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbAddMsgAttribute()
 */
KvaDbStatus WINAPI kvaDbDeleteMsgAttribute(KvaDbMessageHnd mh, KvaDbAttributeHnd ah);

/**
 * Add an attribute to a signal.
 *
 * \param[in]  sh    A handle to a signal
 * \param[in]  adh   A handle to an attribute definition
 * \param[out] ah    An attribute handle
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbDeleteSignalAttribute()
 */
KvaDbStatus WINAPI kvaDbAddSignalAttribute(KvaDbSignalHnd sh, KvaDbAttributeDefHnd adh, KvaDbAttributeHnd *ah);

/**
 * Delete an attribute from a signal.
 *
 * \param[in]  sh    A handle to a signal
 * \param[in]  ah    A handle to an attribute
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbAddSignalAttribute()
 */
KvaDbStatus WINAPI kvaDbDeleteSignalAttribute(KvaDbSignalHnd sh, KvaDbAttributeHnd ah);

/**
 * Add an attribute to a signal.
 *
 * \param[in]  nh    A handle to a node
 * \param[in]  adh   A handle to an attribute definition
 * \param[out] ah    An attribute handle
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbDeleteNodeAttribute()
 */
KvaDbStatus WINAPI kvaDbAddNodeAttribute(KvaDbNodeHnd nh, KvaDbAttributeDefHnd adh, KvaDbAttributeHnd *ah);

/**
 * Delete an attribute from a signal.
 *
 * \param[in]  nh    A handle to a node
 * \param[in]  ah    A handle to an attribute
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbAddNodeAttribute()
 */
KvaDbStatus WINAPI kvaDbDeleteNodeAttribute(KvaDbNodeHnd nh, KvaDbAttributeHnd ah);

/**
 * Get a handle to the first attribute for a message.
 *
 * \param[in]  mh   A handle to a message
 * \param[out] ah   An attribute definition handle
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbGetNextAttribute()
 */
KvaDbStatus WINAPI kvaDbGetFirstMsgAttribute(KvaDbMessageHnd mh, KvaDbAttributeHnd *ah);

/**
 * Get a handle to an attribute with specified name for a message.
 *
 * \param[in]  mh   A handle to a message
 * \param[in]  attrName  The attribute name to search for
 * \param[out] ah  An attribute definition handle
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 */
KvaDbStatus WINAPI kvaDbGetMsgAttributeByName(KvaDbMessageHnd mh, const char *attrName, KvaDbAttributeHnd *ah);

/**
 * Get a handle to the first attribute for a signal.
 *
 * \param[in]  sh   A handle to a signal
 * \param[out] ah   An attribute definition handle
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbGetNextAttribute()
 */
KvaDbStatus WINAPI kvaDbGetFirstSignalAttribute(KvaDbSignalHnd sh, KvaDbAttributeHnd *ah);

/**
 * Get a handle to an attribute with specified name for a signal.
 *
 * \param[in]  sh   A handle to a signal
 * \param[in]  attrName  The attribute name to search for
 * \param[out] ah  An attribute definition handle
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 */
KvaDbStatus WINAPI kvaDbGetSignalAttributeByName(KvaDbSignalHnd sh, const char *attrName, KvaDbAttributeHnd *ah);

/**
 * Get a handle to the first attribute for a node.
 *
 * \param[in]  nh   A handle to a node
 * \param[out] ah   An attribute definition handle
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbGetNextAttribute()
 */
KvaDbStatus WINAPI kvaDbGetFirstNodeAttribute(KvaDbNodeHnd nh, KvaDbAttributeHnd *ah);

/**
 * Get a handle to an attribute with specified name for a node.
 *
 * \param[in]  nh   A handle to a node
 * \param[in]  attrName  The attribute name to search for
 * \param[out] ah  An attribute definition handle
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 */
KvaDbStatus WINAPI kvaDbGetNodeAttributeByName(KvaDbNodeHnd nh, const char *attrName, KvaDbAttributeHnd *ah);

/**
 * Get a handle to the following attribute.
 *
 * \param[in]  ah    A handle to an attribute
 * \param[out] nah   An attribute handle
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbGetFirstMsgAttribute()
 * \sa kvaDbGetFirstSignalAttribute()
 * \sa kvaDbGetFirstNodeAttribute()
 */
KvaDbStatus WINAPI kvaDbGetNextAttribute(KvaDbAttributeHnd ah, KvaDbAttributeHnd *nah);

/**
 * Get the type for an attribute.
 *
 * \param[in]  ah   A handle to an attribute
 * \param[out] at   The type of the attribute
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbGetAttributeName()
 */
KvaDbStatus WINAPI kvaDbGetAttributeType(KvaDbAttributeHnd ah, KvaDbAttributeType *at);

/**
 * Get the name for an attribute.
 *
 * \param[in]  ah      A handle to an attribute
 * \param[out] buf     The buffer that will hold the attribute name
 * \param[in]  buflen  The length of the buffer.
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbGetAttributeType()
 */
KvaDbStatus WINAPI kvaDbGetAttributeName(KvaDbAttributeHnd adh, char *buf, size_t buflen);

/**
 * Set the value for an attribute of type integer.
 *
 * \param[in]  ah   A handle to an attribute
 * \param[in]  val  The value to set
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbGetAttributeValueInt()
 */
KvaDbStatus WINAPI kvaDbSetAttributeValueInt(KvaDbAttributeHnd ah, int val);

/**
 * Get the value for an attribute of type integer.
 *
 * \param[in]  ah   A handle to an attribute
 * \param[out] val  The value of the attribute
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbSetAttributeValueInt()
 */
KvaDbStatus WINAPI kvaDbGetAttributeValueInt(KvaDbAttributeHnd ah, int *val);

/**
 * Set the value for an attribute of type float.
 *
 * \param[in]  ah   A handle to an attribute
 * \param[in]  val  The value to set
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbGetAttributeValueFloat()
 */
KvaDbStatus WINAPI kvaDbSetAttributeValueFloat(KvaDbAttributeHnd ah, float val);

/**
 * Get the value for an attribute of type float.
 *
 * \param[in]  ah   A handle to an attribute
 * \param[out] val  The value of the attribute
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbSetAttributeValueFloat()
 */
KvaDbStatus WINAPI kvaDbGetAttributeValueFloat(KvaDbAttributeHnd ah, float *val);

/**
 * Set the value for an attribute of type string.
 *
 * \param[in]  ah   A handle to an attribute
 * \param[in] buf     The buffer that holds the attribute value
 * \param[in]  buflen  The length of the buffer.
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbGetAttributeValueString()
 */
KvaDbStatus WINAPI kvaDbSetAttributeValueString(KvaDbAttributeHnd ah, const char *buf, int buflen);

/**
 * Get the value for an attribute of type string.
 *
 * \param[in]  ah   A handle to an attribute
 * \param[out] buf     The buffer that holds the attribute value
 * \param[out]  buflen  The length of the buffer.
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbSetAttributeValueString()
 */
KvaDbStatus WINAPI kvaDbGetAttributeValueString(KvaDbAttributeHnd ah, char *buf, size_t buflen);

/**
 * Set the value for an attribute of type enumeration.
 *
 * \param[in]  ah   A handle to an attribute
 * \param[in]  val  The value to set
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbGetAttributeValueEnumeration()
 */
KvaDbStatus WINAPI kvaDbSetAttributeValueEnumeration(KvaDbAttributeHnd ah, int val);

/**
 * Get the value for an attribute of type enumeration.
 *
 * \param[in]  ah   A handle to an attribute
 * \param[out] val  The value of the attribute
 *
 * \return ::kvaDbOK (zero) if success
 * \return \ref kvaDbErr_xxx (negative) if failure
 *
 * \sa kvaDbSetAttributeValueEnumeration()
 */
KvaDbStatus WINAPI kvaDbGetAttributeValueEnumeration(KvaDbAttributeHnd ah, int *val);



#ifdef __cplusplus
}
#endif

/** @} */

#endif //KVADBLIB_H
